<?php
/**
 * Complete Database Setup for Production
 * Run this once: https://4102009823.ceiscy.com/school/database/setup_complete_database.php
 * This will create all missing tables and columns
 */

require_once __DIR__ . '/../config/database.php';

header('Content-Type: text/html; charset=utf-8');
?>
<!DOCTYPE html>
<html>
<head>
    <title>Database Setup - Divine Grace School</title>
    <style>
        body { font-family: Arial, sans-serif; max-width: 1200px; margin: 20px auto; padding: 20px; }
        .success { color: green; padding: 10px; background: #e8f5e9; margin: 5px 0; border-radius: 4px; }
        .error { color: red; padding: 10px; background: #ffebee; margin: 5px 0; border-radius: 4px; }
        .info { color: blue; padding: 10px; background: #e3f2fd; margin: 5px 0; border-radius: 4px; }
        .warning { color: orange; padding: 10px; background: #fff3e0; margin: 5px 0; border-radius: 4px; }
        h2 { color: #333; border-bottom: 2px solid #4CAF50; padding-bottom: 10px; }
        pre { background: #f5f5f5; padding: 10px; border-radius: 4px; overflow-x: auto; }
    </style>
</head>
<body>
    <h1>🚀 Divine Grace School - Database Setup</h1>
    
<?php

try {
    $pdo = getDBConnection();
    echo "<div class='success'>✅ Database connection successful!</div>";
    
    // 1. Add feeding columns to students table
    echo "<h2>1. Feeding System Columns</h2>";
    
    $feedingColumns = [
        'feeding_balance' => "DECIMAL(10,2) DEFAULT 0.00",
        'feeding_weekly_balance' => "DECIMAL(10,2) DEFAULT 0.00",
        'feeding_payment_frequency' => "ENUM('daily', 'weekly', 'monthly', 'termly') DEFAULT 'daily'",
        'custom_feeding_fee' => "DECIMAL(10,2) NULL",
        'feeding_scholarship_percentage' => "INT DEFAULT 0",
        'feeding_notes' => "TEXT NULL"
    ];
    
    foreach ($feedingColumns as $column => $definition) {
        $check = $pdo->query("SHOW COLUMNS FROM students LIKE '$column'")->fetch();
        if (!$check) {
            $pdo->exec("ALTER TABLE students ADD COLUMN $column $definition");
            echo "<div class='success'>✅ Added column: $column</div>";
        } else {
            echo "<div class='info'>✓ Column exists: $column</div>";
        }
    }
    
    // 2. Create feeding_records table if not exists
    echo "<h2>2. Feeding Records Table</h2>";
    
    $createFeedingRecords = "
    CREATE TABLE IF NOT EXISTS feeding_records (
        id INT AUTO_INCREMENT PRIMARY KEY,
        student_id INT NOT NULL,
        date DATE NOT NULL,
        status ENUM('present', 'absent') DEFAULT 'present',
        amount_paid DECIMAL(10,2) DEFAULT 0.00,
        payment_method ENUM('cash', 'bank_transfer', 'mobile_money', 'cheque') NULL,
        notes TEXT NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        FOREIGN KEY (student_id) REFERENCES students(id) ON DELETE CASCADE,
        UNIQUE KEY unique_student_date (student_id, date)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    
    $pdo->exec($createFeedingRecords);
    echo "<div class='success'>✅ feeding_records table created</div>";
    
    // Add notes column if table exists but column is missing
    $checkNotesColumn = $pdo->query("SHOW COLUMNS FROM feeding_records LIKE 'notes'")->fetch();
    if (!$checkNotesColumn) {
        $pdo->exec("ALTER TABLE feeding_records ADD COLUMN notes TEXT NULL AFTER payment_method");
        echo "<div class='success'>✅ Added notes column to feeding_records</div>";
    } else {
        echo "<div class='info'>✓ notes column exists in feeding_records</div>";
    }
    
    // 3. Create academic tables
    echo "<h2>3. Academic System Tables</h2>";
    
    // Terms table
    $createTerms = "
    CREATE TABLE IF NOT EXISTS terms (
        id INT AUTO_INCREMENT PRIMARY KEY,
        name VARCHAR(100) NOT NULL,
        start_date DATE NOT NULL,
        end_date DATE NOT NULL,
        academic_year VARCHAR(20) NOT NULL,
        is_current BOOLEAN DEFAULT FALSE,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    
    $pdo->exec($createTerms);
    echo "<div class='success'>✅ terms table ready</div>";
    
    // Classes table
    $createClasses = "
    CREATE TABLE IF NOT EXISTS classes (
        id INT AUTO_INCREMENT PRIMARY KEY,
        name VARCHAR(100) NOT NULL UNIQUE,
        level VARCHAR(50) NULL,
        capacity INT DEFAULT 30,
        class_teacher_id INT NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        FOREIGN KEY (class_teacher_id) REFERENCES users(id) ON DELETE SET NULL
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    
    $pdo->exec($createClasses);
    echo "<div class='success'>✅ classes table ready</div>";
    
    // Subjects table
    $createSubjects = "
    CREATE TABLE IF NOT EXISTS subjects (
        id INT AUTO_INCREMENT PRIMARY KEY,
        name VARCHAR(100) NOT NULL,
        code VARCHAR(20) UNIQUE,
        description TEXT NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    
    $pdo->exec($createSubjects);
    echo "<div class='success'>✅ subjects table ready</div>";
    
    // Assessments table
    $createAssessments = "
    CREATE TABLE IF NOT EXISTS assessments (
        id INT AUTO_INCREMENT PRIMARY KEY,
        term_id INT NOT NULL,
        class_id INT NOT NULL,
        subject_id INT NOT NULL,
        student_id INT NOT NULL,
        assessment_type ENUM('quiz', 'assignment', 'midterm', 'final', 'project') NOT NULL,
        score DECIMAL(5,2) NOT NULL,
        max_score DECIMAL(5,2) DEFAULT 100.00,
        date DATE NOT NULL,
        remarks TEXT NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        FOREIGN KEY (term_id) REFERENCES terms(id) ON DELETE CASCADE,
        FOREIGN KEY (class_id) REFERENCES classes(id) ON DELETE CASCADE,
        FOREIGN KEY (subject_id) REFERENCES subjects(id) ON DELETE CASCADE,
        FOREIGN KEY (student_id) REFERENCES students(id) ON DELETE CASCADE
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    
    $pdo->exec($createAssessments);
    echo "<div class='success'>✅ assessments table ready</div>";
    
    // Attendance table
    $createAttendance = "
    CREATE TABLE IF NOT EXISTS attendance (
        id INT AUTO_INCREMENT PRIMARY KEY,
        student_id INT NOT NULL,
        class_id INT NOT NULL,
        date DATE NOT NULL,
        status ENUM('present', 'absent', 'late', 'excused') DEFAULT 'present',
        remarks TEXT NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        FOREIGN KEY (student_id) REFERENCES students(id) ON DELETE CASCADE,
        FOREIGN KEY (class_id) REFERENCES classes(id) ON DELETE CASCADE,
        UNIQUE KEY unique_student_date (student_id, date)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    
    $pdo->exec($createAttendance);
    echo "<div class='success'>✅ attendance table ready</div>";
    
    // 4. Add class_id to students if missing
    echo "<h2>4. Student-Class Relationship</h2>";
    
    $checkClassId = $pdo->query("SHOW COLUMNS FROM students LIKE 'class_id'")->fetch();
    if (!$checkClassId) {
        $pdo->exec("ALTER TABLE students ADD COLUMN class_id INT NULL AFTER class");
        $pdo->exec("ALTER TABLE students ADD FOREIGN KEY (class_id) REFERENCES classes(id) ON DELETE SET NULL");
        echo "<div class='success'>✅ Added class_id column to students</div>";
    } else {
        echo "<div class='info'>✓ class_id column exists</div>";
    }
    
    // 5. Create transport tables
    echo "<h2>5. Transport System Tables</h2>";
    
    // Routes table
    $createRoutes = "
    CREATE TABLE IF NOT EXISTS transport_routes (
        id INT AUTO_INCREMENT PRIMARY KEY,
        name VARCHAR(100) NOT NULL,
        description TEXT NULL,
        fee DECIMAL(10,2) NOT NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    
    $pdo->exec($createRoutes);
    echo "<div class='success'>✅ transport_routes table ready</div>";
    
    // Vehicles table
    $createVehicles = "
    CREATE TABLE IF NOT EXISTS transport_vehicles (
        id INT AUTO_INCREMENT PRIMARY KEY,
        registration_number VARCHAR(50) NOT NULL UNIQUE,
        model VARCHAR(100) NOT NULL,
        capacity INT NOT NULL,
        driver_name VARCHAR(100) NULL,
        driver_phone VARCHAR(20) NULL,
        status ENUM('active', 'maintenance', 'inactive') DEFAULT 'active',
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    
    $pdo->exec($createVehicles);
    echo "<div class='success'>✅ transport_vehicles table ready</div>";
    
    // 6. Summary
    echo "<h2>6. Database Summary</h2>";
    
    $tables = $pdo->query("SHOW TABLES")->fetchAll(PDO::FETCH_COLUMN);
    echo "<div class='success'>✅ Total tables: " . count($tables) . "</div>";
    echo "<pre>" . implode("\n", $tables) . "</pre>";
    
    // Check students table structure
    $studentColumns = $pdo->query("SHOW COLUMNS FROM students")->fetchAll(PDO::FETCH_ASSOC);
    echo "<h3>Students Table Columns (" . count($studentColumns) . ")</h3>";
    echo "<pre>";
    foreach ($studentColumns as $col) {
        echo $col['Field'] . " - " . $col['Type'] . "\n";
    }
    echo "</pre>";
    
    echo "<div class='success' style='font-size: 18px; margin-top: 30px;'>
        🎉 <strong>Database setup complete!</strong><br>
        Your system is ready to use.
    </div>";
    
    echo "<div class='info'>
        <strong>Next Steps:</strong><br>
        1. Go back to your application: <a href='../'>https://4102009823.ceiscy.com/school/</a><br>
        2. Login and test the feeding system<br>
        3. All features should now work correctly!
    </div>";
    
} catch (Exception $e) {
    echo "<div class='error'>❌ Error: " . htmlspecialchars($e->getMessage()) . "</div>";
    echo "<pre>" . htmlspecialchars($e->getTraceAsString()) . "</pre>";
}

?>
</body>
</html>
