<?php
/**
 * Fix feeding_records Table Structure
 * Run this: https://4102009823.ceiscy.com/school/database/fix_feeding_records_table.php
 */

require_once __DIR__ . '/../config/database.php';

header('Content-Type: text/html; charset=utf-8');
?>
<!DOCTYPE html>
<html>
<head>
    <title>Fix Feeding Records Table</title>
    <style>
        body { font-family: Arial, sans-serif; max-width: 900px; margin: 20px auto; padding: 20px; background: #f5f5f5; }
        .container { background: white; padding: 30px; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.1); }
        .success { color: #4CAF50; padding: 10px; background: #e8f5e9; margin: 5px 0; border-radius: 4px; border-left: 4px solid #4CAF50; }
        .error { color: #f44336; padding: 10px; background: #ffebee; margin: 5px 0; border-radius: 4px; border-left: 4px solid #f44336; }
        .info { color: #2196F3; padding: 10px; background: #e3f2fd; margin: 5px 0; border-radius: 4px; border-left: 4px solid #2196F3; }
        .warning { color: #ff9800; padding: 10px; background: #fff3e0; margin: 5px 0; border-radius: 4px; border-left: 4px solid #ff9800; }
        h1 { color: #333; margin-top: 0; }
        h2 { color: #666; border-bottom: 2px solid #4CAF50; padding-bottom: 10px; margin-top: 30px; }
        pre { background: #f5f5f5; padding: 10px; border-radius: 4px; overflow-x: auto; font-size: 12px; }
        .btn { display: inline-block; padding: 12px 24px; background: #4CAF50; color: white; text-decoration: none; border-radius: 4px; margin-top: 20px; }
    </style>
</head>
<body>
    <div class="container">
        <h1>🔧 Fix Feeding Records Table</h1>
        
<?php

try {
    $pdo = getDBConnection();
    echo "<div class='success'>✅ Database connected</div>";
    
    // Check current structure
    echo "<h2>Current Table Structure</h2>";
    $currentColumns = $pdo->query("SHOW COLUMNS FROM feeding_records")->fetchAll(PDO::FETCH_ASSOC);
    
    echo "<pre>";
    foreach ($currentColumns as $col) {
        echo $col['Field'] . " | " . $col['Type'] . " | " . $col['Null'] . " | " . $col['Default'] . "\n";
    }
    echo "</pre>";
    
    $existingCols = array_column($currentColumns, 'Field');
    
    // Define required columns
    $requiredColumns = [
        'id' => ['type' => 'INT AUTO_INCREMENT PRIMARY KEY', 'after' => null],
        'student_id' => ['type' => 'INT NOT NULL', 'after' => 'id'],
        'date' => ['type' => 'DATE NOT NULL', 'after' => 'student_id'],
        'status' => ['type' => "ENUM('present', 'absent') DEFAULT 'present'", 'after' => 'date'],
        'amount_paid' => ['type' => 'DECIMAL(10,2) DEFAULT 0.00', 'after' => 'status'],
        'payment_method' => ['type' => "ENUM('cash', 'bank_transfer', 'mobile_money', 'cheque') NULL", 'after' => 'amount_paid'],
        'notes' => ['type' => 'TEXT NULL', 'after' => 'payment_method'],
        'created_at' => ['type' => 'TIMESTAMP DEFAULT CURRENT_TIMESTAMP', 'after' => 'notes'],
        'updated_at' => ['type' => 'TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP', 'after' => 'created_at']
    ];
    
    echo "<h2>Adding Missing Columns</h2>";
    
    $added = [];
    $errors = [];
    
    foreach ($requiredColumns as $colName => $colDef) {
        if (!in_array($colName, $existingCols)) {
            try {
                $sql = "ALTER TABLE feeding_records ADD COLUMN $colName " . $colDef['type'];
                if ($colDef['after']) {
                    $sql .= " AFTER " . $colDef['after'];
                }
                
                $pdo->exec($sql);
                echo "<div class='success'>✅ Added column: $colName</div>";
                $added[] = $colName;
            } catch (Exception $e) {
                echo "<div class='error'>❌ Error adding $colName: " . $e->getMessage() . "</div>";
                $errors[] = $colName . ": " . $e->getMessage();
            }
        } else {
            echo "<div class='info'>✓ Column exists: $colName</div>";
        }
    }
    
    // Add foreign key if not exists
    echo "<h2>Foreign Keys</h2>";
    try {
        $fks = $pdo->query("
            SELECT CONSTRAINT_NAME 
            FROM information_schema.KEY_COLUMN_USAGE 
            WHERE TABLE_NAME = 'feeding_records' 
            AND CONSTRAINT_NAME LIKE 'feeding_records_ibfk%'
        ")->fetchAll(PDO::FETCH_COLUMN);
        
        if (empty($fks)) {
            $pdo->exec("ALTER TABLE feeding_records ADD FOREIGN KEY (student_id) REFERENCES students(id) ON DELETE CASCADE");
            echo "<div class='success'>✅ Added foreign key constraint</div>";
        } else {
            echo "<div class='info'>✓ Foreign key exists</div>";
        }
    } catch (Exception $e) {
        echo "<div class='warning'>⚠️ Foreign key: " . $e->getMessage() . "</div>";
    }
    
    // Add unique constraint if not exists
    echo "<h2>Unique Constraints</h2>";
    try {
        $indexes = $pdo->query("SHOW INDEX FROM feeding_records WHERE Key_name = 'unique_student_date'")->fetchAll();
        
        if (empty($indexes)) {
            $pdo->exec("ALTER TABLE feeding_records ADD UNIQUE KEY unique_student_date (student_id, date)");
            echo "<div class='success'>✅ Added unique constraint</div>";
        } else {
            echo "<div class='info'>✓ Unique constraint exists</div>";
        }
    } catch (Exception $e) {
        echo "<div class='warning'>⚠️ Unique constraint: " . $e->getMessage() . "</div>";
    }
    
    // Final verification
    echo "<h2>Final Structure</h2>";
    $finalColumns = $pdo->query("SHOW COLUMNS FROM feeding_records")->fetchAll(PDO::FETCH_ASSOC);
    
    echo "<pre>";
    foreach ($finalColumns as $col) {
        echo $col['Field'] . " | " . $col['Type'] . "\n";
    }
    echo "</pre>";
    
    $finalColNames = array_column($finalColumns, 'Field');
    $missingCols = array_diff(array_keys($requiredColumns), $finalColNames);
    
    echo "<h2>Summary</h2>";
    
    if (count($added) > 0) {
        echo "<div class='success'><strong>✅ Added " . count($added) . " columns:</strong><br>";
        echo implode(', ', $added);
        echo "</div>";
    }
    
    if (count($errors) > 0) {
        echo "<div class='error'><strong>❌ Errors:</strong><br>";
        foreach ($errors as $error) {
            echo "• $error<br>";
        }
        echo "</div>";
    }
    
    if (empty($missingCols) && count($errors) === 0) {
        echo "<div class='success' style='font-size: 18px; padding: 20px;'>
            🎉 <strong>Perfect!</strong><br><br>
            The feeding_records table is now properly configured.<br>
            All required columns are present.
        </div>";
        
        echo "<a href='../' class='btn'>← Back to Application</a>";
    } else if (!empty($missingCols)) {
        echo "<div class='error'>
            <strong>Still missing columns:</strong><br>
            " . implode(', ', $missingCols) . "
        </div>";
    }
    
} catch (Exception $e) {
    echo "<div class='error'>
        <strong>❌ Fatal Error:</strong><br>
        " . htmlspecialchars($e->getMessage()) . "
    </div>";
    echo "<pre>" . htmlspecialchars($e->getTraceAsString()) . "</pre>";
}

?>
    </div>
</body>
</html>
