-- Transport System Database Schema
-- Divine Grace School Management System

-- 1. Transport Routes Table
CREATE TABLE IF NOT EXISTS transport_routes (
    route_id INT AUTO_INCREMENT PRIMARY KEY,
    route_name VARCHAR(100) NOT NULL,
    route_number VARCHAR(50) UNIQUE NOT NULL,
    description TEXT,
    total_distance DECIMAL(10,2) DEFAULT 0, -- in kilometers
    estimated_duration INT DEFAULT 0, -- in minutes
    base_fee DECIMAL(10,2) DEFAULT 0,
    status ENUM('active', 'inactive') DEFAULT 'active',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_route_number (route_number),
    INDEX idx_status (status)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- 2. Route Stops Table
CREATE TABLE IF NOT EXISTS route_stops (
    stop_id INT AUTO_INCREMENT PRIMARY KEY,
    route_id INT NOT NULL,
    stop_name VARCHAR(200) NOT NULL,
    stop_order INT NOT NULL, -- sequence of stops
    distance_from_school DECIMAL(10,2) DEFAULT 0, -- in kilometers
    pickup_time TIME,
    dropoff_time TIME,
    landmark TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (route_id) REFERENCES transport_routes(route_id) ON DELETE CASCADE,
    INDEX idx_route_id (route_id),
    INDEX idx_stop_order (stop_order)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- 3. Vehicles Table
CREATE TABLE IF NOT EXISTS transport_vehicles (
    vehicle_id INT AUTO_INCREMENT PRIMARY KEY,
    vehicle_number VARCHAR(50) UNIQUE NOT NULL,
    vehicle_type ENUM('bus', 'van', 'car') DEFAULT 'bus',
    capacity INT NOT NULL,
    model VARCHAR(100),
    year INT,
    registration_number VARCHAR(100),
    insurance_expiry DATE,
    license_expiry DATE,
    last_maintenance_date DATE,
    next_maintenance_date DATE,
    fuel_type ENUM('petrol', 'diesel', 'electric', 'hybrid') DEFAULT 'diesel',
    average_fuel_consumption DECIMAL(10,2), -- liters per 100km
    status ENUM('active', 'maintenance', 'inactive') DEFAULT 'active',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_vehicle_number (vehicle_number),
    INDEX idx_status (status)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- 4. Drivers Table
CREATE TABLE IF NOT EXISTS transport_drivers (
    driver_id INT AUTO_INCREMENT PRIMARY KEY,
    driver_name VARCHAR(200) NOT NULL,
    phone VARCHAR(20),
    email VARCHAR(100),
    license_number VARCHAR(100) UNIQUE,
    license_expiry DATE,
    date_of_birth DATE,
    address TEXT,
    emergency_contact VARCHAR(20),
    status ENUM('active', 'inactive', 'suspended') DEFAULT 'active',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_license_number (license_number),
    INDEX idx_status (status)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- 5. Route Assignments Table (Vehicle + Driver + Route)
CREATE TABLE IF NOT EXISTS route_assignments (
    assignment_id INT AUTO_INCREMENT PRIMARY KEY,
    route_id INT NOT NULL,
    vehicle_id INT NOT NULL,
    driver_id INT NOT NULL,
    shift ENUM('morning', 'afternoon', 'both') DEFAULT 'both',
    effective_from DATE NOT NULL,
    effective_to DATE,
    status ENUM('active', 'inactive') DEFAULT 'active',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (route_id) REFERENCES transport_routes(route_id) ON DELETE CASCADE,
    FOREIGN KEY (vehicle_id) REFERENCES transport_vehicles(vehicle_id) ON DELETE CASCADE,
    FOREIGN KEY (driver_id) REFERENCES transport_drivers(driver_id) ON DELETE CASCADE,
    INDEX idx_route_id (route_id),
    INDEX idx_vehicle_id (vehicle_id),
    INDEX idx_driver_id (driver_id),
    INDEX idx_status (status)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- 6. Student Transport Registration
CREATE TABLE IF NOT EXISTS student_transport (
    transport_registration_id INT AUTO_INCREMENT PRIMARY KEY,
    student_id VARCHAR(50) NOT NULL,
    route_id INT NOT NULL,
    stop_id INT NOT NULL,
    transport_fee DECIMAL(10,2) NOT NULL,
    discount_percentage DECIMAL(5,2) DEFAULT 0,
    final_fee DECIMAL(10,2) NOT NULL,
    billing_cycle ENUM('monthly', 'termly', 'yearly') DEFAULT 'termly',
    start_date DATE NOT NULL,
    end_date DATE,
    status ENUM('active', 'inactive', 'suspended') DEFAULT 'active',
    special_requirements TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (student_id) REFERENCES students(student_id) ON DELETE CASCADE,
    FOREIGN KEY (route_id) REFERENCES transport_routes(route_id) ON DELETE CASCADE,
    FOREIGN KEY (stop_id) REFERENCES route_stops(stop_id) ON DELETE CASCADE,
    INDEX idx_student_id (student_id),
    INDEX idx_route_id (route_id),
    INDEX idx_status (status)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- 7. Daily Transport Attendance
CREATE TABLE IF NOT EXISTS transport_attendance (
    attendance_id INT AUTO_INCREMENT PRIMARY KEY,
    student_id VARCHAR(50) NOT NULL,
    route_id INT NOT NULL,
    vehicle_id INT NOT NULL,
    attendance_date DATE NOT NULL,
    shift ENUM('morning', 'afternoon') NOT NULL,
    boarded BOOLEAN DEFAULT FALSE,
    boarding_time TIME,
    alighted BOOLEAN DEFAULT FALSE,
    alighting_time TIME,
    stop_id INT,
    recorded_by VARCHAR(100),
    notes TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (student_id) REFERENCES students(student_id) ON DELETE CASCADE,
    FOREIGN KEY (route_id) REFERENCES transport_routes(route_id) ON DELETE CASCADE,
    FOREIGN KEY (vehicle_id) REFERENCES transport_vehicles(vehicle_id) ON DELETE CASCADE,
    FOREIGN KEY (stop_id) REFERENCES route_stops(stop_id) ON DELETE SET NULL,
    UNIQUE KEY unique_attendance (student_id, attendance_date, shift),
    INDEX idx_student_id (student_id),
    INDEX idx_date (attendance_date),
    INDEX idx_route_id (route_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- 8. Transport Fee Payments
CREATE TABLE IF NOT EXISTS transport_payments (
    payment_id INT AUTO_INCREMENT PRIMARY KEY,
    student_id VARCHAR(50) NOT NULL,
    transport_registration_id INT NOT NULL,
    amount_paid DECIMAL(10,2) NOT NULL,
    payment_date DATE NOT NULL,
    payment_method ENUM('cash', 'bank_transfer', 'mobile_money', 'cheque', 'card') DEFAULT 'cash',
    period_from DATE NOT NULL,
    period_to DATE NOT NULL,
    balance DECIMAL(10,2) DEFAULT 0,
    receipt_number VARCHAR(100) UNIQUE,
    recorded_by VARCHAR(100),
    notes TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (student_id) REFERENCES students(student_id) ON DELETE CASCADE,
    FOREIGN KEY (transport_registration_id) REFERENCES student_transport(transport_registration_id) ON DELETE CASCADE,
    INDEX idx_student_id (student_id),
    INDEX idx_payment_date (payment_date),
    INDEX idx_receipt_number (receipt_number)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- 9. Fuel Consumption Logs
CREATE TABLE IF NOT EXISTS fuel_logs (
    fuel_log_id INT AUTO_INCREMENT PRIMARY KEY,
    vehicle_id INT NOT NULL,
    log_date DATE NOT NULL,
    odometer_reading DECIMAL(10,2) NOT NULL, -- current reading in km
    fuel_quantity DECIMAL(10,2) NOT NULL, -- liters
    fuel_cost DECIMAL(10,2) NOT NULL,
    fuel_type ENUM('petrol', 'diesel', 'electric', 'hybrid') NOT NULL,
    station_name VARCHAR(200),
    distance_covered DECIMAL(10,2), -- calculated from previous reading
    fuel_efficiency DECIMAL(10,2), -- km per liter (calculated)
    filled_by VARCHAR(100),
    receipt_number VARCHAR(100),
    notes TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (vehicle_id) REFERENCES transport_vehicles(vehicle_id) ON DELETE CASCADE,
    INDEX idx_vehicle_id (vehicle_id),
    INDEX idx_log_date (log_date)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- 10. Trip Logs (Daily trips for fuel calculation)
CREATE TABLE IF NOT EXISTS trip_logs (
    trip_id INT AUTO_INCREMENT PRIMARY KEY,
    vehicle_id INT NOT NULL,
    driver_id INT NOT NULL,
    route_id INT NOT NULL,
    trip_date DATE NOT NULL,
    shift ENUM('morning', 'afternoon') NOT NULL,
    start_time TIME,
    end_time TIME,
    start_odometer DECIMAL(10,2),
    end_odometer DECIMAL(10,2),
    distance_covered DECIMAL(10,2), -- calculated
    students_transported INT DEFAULT 0,
    fuel_consumed DECIMAL(10,2), -- estimated based on distance and vehicle efficiency
    status ENUM('completed', 'cancelled', 'in_progress') DEFAULT 'completed',
    notes TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (vehicle_id) REFERENCES transport_vehicles(vehicle_id) ON DELETE CASCADE,
    FOREIGN KEY (driver_id) REFERENCES transport_drivers(driver_id) ON DELETE CASCADE,
    FOREIGN KEY (route_id) REFERENCES transport_routes(route_id) ON DELETE CASCADE,
    INDEX idx_vehicle_id (vehicle_id),
    INDEX idx_trip_date (trip_date),
    INDEX idx_route_id (route_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Insert sample data for testing
INSERT INTO transport_routes (route_name, route_number, description, total_distance, estimated_duration, base_fee) VALUES
('City Center Route', 'RT-001', 'Covers main city center areas', 15.5, 45, 150.00),
('Suburb North Route', 'RT-002', 'Northern suburbs coverage', 22.0, 60, 200.00),
('Suburb South Route', 'RT-003', 'Southern suburbs coverage', 18.5, 50, 175.00);

INSERT INTO transport_vehicles (vehicle_number, vehicle_type, capacity, model, year, fuel_type, average_fuel_consumption) VALUES
('BUS-001', 'bus', 45, 'Toyota Coaster', 2020, 'diesel', 12.5),
('VAN-001', 'van', 15, 'Toyota Hiace', 2021, 'diesel', 10.0),
('BUS-002', 'bus', 50, 'Mercedes Sprinter', 2019, 'diesel', 13.0);

INSERT INTO transport_drivers (driver_name, phone, license_number, license_expiry) VALUES
('John Mensah', '0244123456', 'DL-2024-001', '2026-12-31'),
('Kwame Asante', '0244234567', 'DL-2024-002', '2027-06-30'),
('Ama Serwaa', '0244345678', 'DL-2024-003', '2026-09-30');
