-- Academic System Database Schema
-- Phase 1: Classes, Subjects, Grading, Assessments, Attendance, Report Cards

-- ============================================
-- 1. CLASSES & SUBJECTS MANAGEMENT
-- ============================================

-- Academic Terms/Semesters
CREATE TABLE IF NOT EXISTS academic_terms (
    id INT AUTO_INCREMENT PRIMARY KEY,
    term_name VARCHAR(100) NOT NULL,
    academic_year VARCHAR(20) NOT NULL,
    start_date DATE NOT NULL,
    end_date DATE NOT NULL,
    is_current BOOLEAN DEFAULT FALSE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_academic_year (academic_year),
    INDEX idx_is_current (is_current)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Classes (KG1, Class 1-6, etc.)
CREATE TABLE IF NOT EXISTS classes (
    id INT AUTO_INCREMENT PRIMARY KEY,
    class_name VARCHAR(50) NOT NULL UNIQUE,
    class_level INT NOT NULL,
    class_teacher_id INT NULL,
    capacity INT DEFAULT 30,
    description TEXT,
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_class_level (class_level),
    INDEX idx_is_active (is_active)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Subjects
CREATE TABLE IF NOT EXISTS subjects (
    id INT AUTO_INCREMENT PRIMARY KEY,
    subject_name VARCHAR(100) NOT NULL,
    subject_code VARCHAR(20) UNIQUE,
    description TEXT,
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_subject_code (subject_code)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Class-Subject Mapping (which subjects are taught in which classes)
CREATE TABLE IF NOT EXISTS class_subjects (
    id INT AUTO_INCREMENT PRIMARY KEY,
    class_id INT NOT NULL,
    subject_id INT NOT NULL,
    teacher_id INT NULL,
    credit_hours INT DEFAULT 1,
    pass_mark DECIMAL(5,2) DEFAULT 50.00,
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (class_id) REFERENCES classes(id) ON DELETE CASCADE,
    FOREIGN KEY (subject_id) REFERENCES subjects(id) ON DELETE CASCADE,
    UNIQUE KEY unique_class_subject (class_id, subject_id),
    INDEX idx_class_id (class_id),
    INDEX idx_subject_id (subject_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- 2. GRADING & ASSESSMENT SYSTEM
-- ============================================

-- Assessment Types (Test, Exam, Assignment, etc.)
CREATE TABLE IF NOT EXISTS assessment_types (
    id INT AUTO_INCREMENT PRIMARY KEY,
    type_name VARCHAR(50) NOT NULL,
    description TEXT,
    max_score DECIMAL(5,2) DEFAULT 100.00,
    weight_percentage DECIMAL(5,2) DEFAULT 100.00,
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Assessments (specific test/exam instances)
CREATE TABLE IF NOT EXISTS assessments (
    id INT AUTO_INCREMENT PRIMARY KEY,
    term_id INT NOT NULL,
    class_id INT NOT NULL,
    subject_id INT NOT NULL,
    assessment_type_id INT NOT NULL,
    assessment_name VARCHAR(200) NOT NULL,
    assessment_date DATE NOT NULL,
    max_score DECIMAL(5,2) DEFAULT 100.00,
    weight_percentage DECIMAL(5,2) DEFAULT 100.00,
    description TEXT,
    created_by INT NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (term_id) REFERENCES academic_terms(id) ON DELETE CASCADE,
    FOREIGN KEY (class_id) REFERENCES classes(id) ON DELETE CASCADE,
    FOREIGN KEY (subject_id) REFERENCES subjects(id) ON DELETE CASCADE,
    FOREIGN KEY (assessment_type_id) REFERENCES assessment_types(id),
    INDEX idx_term_class_subject (term_id, class_id, subject_id),
    INDEX idx_assessment_date (assessment_date)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Student Scores
CREATE TABLE IF NOT EXISTS student_scores (
    id INT AUTO_INCREMENT PRIMARY KEY,
    assessment_id INT NOT NULL,
    student_id INT NOT NULL,
    score DECIMAL(5,2) NOT NULL,
    remarks TEXT,
    recorded_by INT NOT NULL,
    recorded_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (assessment_id) REFERENCES assessments(id) ON DELETE CASCADE,
    FOREIGN KEY (student_id) REFERENCES students(id) ON DELETE CASCADE,
    UNIQUE KEY unique_student_assessment (assessment_id, student_id),
    INDEX idx_student_id (student_id),
    INDEX idx_assessment_id (assessment_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Grading Scale
CREATE TABLE IF NOT EXISTS grading_scale (
    id INT AUTO_INCREMENT PRIMARY KEY,
    grade VARCHAR(5) NOT NULL,
    min_score DECIMAL(5,2) NOT NULL,
    max_score DECIMAL(5,2) NOT NULL,
    grade_point DECIMAL(3,2) DEFAULT 0.00,
    description VARCHAR(50),
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_score_range (min_score, max_score)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- 3. ATTENDANCE MANAGEMENT
-- ============================================

-- Daily Attendance
CREATE TABLE IF NOT EXISTS attendance (
    id INT AUTO_INCREMENT PRIMARY KEY,
    student_id INT NOT NULL,
    class_id INT NOT NULL,
    attendance_date DATE NOT NULL,
    status ENUM('present', 'absent', 'late', 'excused') NOT NULL DEFAULT 'present',
    arrival_time TIME NULL,
    remarks TEXT,
    recorded_by INT NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (student_id) REFERENCES students(id) ON DELETE CASCADE,
    FOREIGN KEY (class_id) REFERENCES classes(id) ON DELETE CASCADE,
    UNIQUE KEY unique_student_date (student_id, attendance_date),
    INDEX idx_attendance_date (attendance_date),
    INDEX idx_student_id (student_id),
    INDEX idx_class_id (class_id),
    INDEX idx_status (status)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Subject-wise Attendance (optional, for secondary schools)
CREATE TABLE IF NOT EXISTS subject_attendance (
    id INT AUTO_INCREMENT PRIMARY KEY,
    student_id INT NOT NULL,
    class_subject_id INT NOT NULL,
    attendance_date DATE NOT NULL,
    period_number INT NOT NULL,
    status ENUM('present', 'absent', 'late') NOT NULL DEFAULT 'present',
    remarks TEXT,
    recorded_by INT NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (student_id) REFERENCES students(id) ON DELETE CASCADE,
    FOREIGN KEY (class_subject_id) REFERENCES class_subjects(id) ON DELETE CASCADE,
    UNIQUE KEY unique_student_subject_date (student_id, class_subject_id, attendance_date, period_number),
    INDEX idx_attendance_date (attendance_date),
    INDEX idx_student_id (student_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- 4. REPORT CARDS
-- ============================================

-- Report Cards
CREATE TABLE IF NOT EXISTS report_cards (
    id INT AUTO_INCREMENT PRIMARY KEY,
    student_id INT NOT NULL,
    term_id INT NOT NULL,
    class_id INT NOT NULL,
    total_score DECIMAL(8,2) DEFAULT 0.00,
    average_score DECIMAL(5,2) DEFAULT 0.00,
    grade VARCHAR(5),
    class_position INT,
    total_students INT,
    attendance_percentage DECIMAL(5,2),
    days_present INT DEFAULT 0,
    days_absent INT DEFAULT 0,
    teacher_comment TEXT,
    principal_comment TEXT,
    generated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    generated_by INT NOT NULL,
    is_published BOOLEAN DEFAULT FALSE,
    published_at TIMESTAMP NULL,
    FOREIGN KEY (student_id) REFERENCES students(id) ON DELETE CASCADE,
    FOREIGN KEY (term_id) REFERENCES academic_terms(id) ON DELETE CASCADE,
    FOREIGN KEY (class_id) REFERENCES classes(id) ON DELETE CASCADE,
    UNIQUE KEY unique_student_term (student_id, term_id),
    INDEX idx_term_class (term_id, class_id),
    INDEX idx_student_id (student_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Report Card Details (subject-wise breakdown)
CREATE TABLE IF NOT EXISTS report_card_details (
    id INT AUTO_INCREMENT PRIMARY KEY,
    report_card_id INT NOT NULL,
    subject_id INT NOT NULL,
    class_average DECIMAL(5,2),
    student_score DECIMAL(5,2),
    grade VARCHAR(5),
    subject_position INT,
    remarks TEXT,
    FOREIGN KEY (report_card_id) REFERENCES report_cards(id) ON DELETE CASCADE,
    FOREIGN KEY (subject_id) REFERENCES subjects(id) ON DELETE CASCADE,
    INDEX idx_report_card_id (report_card_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- INSERT DEFAULT DATA
-- ============================================

-- Default Assessment Types
INSERT INTO assessment_types (type_name, description, max_score, weight_percentage) VALUES
('Class Test', 'Regular class tests', 100.00, 30.00),
('Mid-Term Exam', 'Mid-term examination', 100.00, 30.00),
('Final Exam', 'End of term examination', 100.00, 40.00),
('Assignment', 'Homework and assignments', 100.00, 10.00),
('Project', 'Class projects', 100.00, 20.00);

-- Default Grading Scale
INSERT INTO grading_scale (grade, min_score, max_score, grade_point, description) VALUES
('A+', 90.00, 100.00, 4.00, 'Excellent'),
('A', 80.00, 89.99, 3.75, 'Very Good'),
('B+', 75.00, 79.99, 3.50, 'Good'),
('B', 70.00, 74.99, 3.00, 'Above Average'),
('C+', 65.00, 69.99, 2.50, 'Average'),
('C', 60.00, 64.99, 2.00, 'Pass'),
('D', 50.00, 59.99, 1.00, 'Below Average'),
('F', 0.00, 49.99, 0.00, 'Fail');

-- Default Classes
INSERT INTO classes (class_name, class_level, capacity, description) VALUES
('KG 1', 1, 25, 'Kindergarten 1'),
('KG 2', 2, 25, 'Kindergarten 2'),
('Class 1', 3, 30, 'Primary 1'),
('Class 2', 4, 30, 'Primary 2'),
('Class 3', 5, 30, 'Primary 3'),
('Class 4', 6, 30, 'Primary 4'),
('Class 5', 7, 30, 'Primary 5'),
('Class 6', 8, 30, 'Primary 6');

-- Default Subjects
INSERT INTO subjects (subject_name, subject_code, description) VALUES
('Mathematics', 'MATH', 'Mathematics'),
('English Language', 'ENG', 'English Language'),
('Science', 'SCI', 'General Science'),
('Social Studies', 'SOC', 'Social Studies'),
('Religious & Moral Education', 'RME', 'Religious and Moral Education'),
('Ghanaian Language', 'GHA', 'Ghanaian Language'),
('Creative Arts', 'ART', 'Creative Arts and Design'),
('Physical Education', 'PE', 'Physical Education'),
('ICT', 'ICT', 'Information and Communication Technology'),
('French', 'FRE', 'French Language');
