<?php
/**
 * Test Feeding System Setup
 * Run: https://4102009823.ceiscy.com/school/api/test_feeding.php
 */

header('Content-Type: text/html; charset=utf-8');

require_once __DIR__ . '/../config/database.php';

?>
<!DOCTYPE html>
<html>
<head>
    <title>Test Feeding System</title>
    <style>
        body { font-family: Arial, sans-serif; max-width: 900px; margin: 20px auto; padding: 20px; background: #f5f5f5; }
        .container { background: white; padding: 30px; border-radius: 8px; }
        .success { color: #4CAF50; padding: 10px; background: #e8f5e9; margin: 5px 0; border-radius: 4px; }
        .error { color: #f44336; padding: 10px; background: #ffebee; margin: 5px 0; border-radius: 4px; }
        .info { color: #2196F3; padding: 10px; background: #e3f2fd; margin: 5px 0; border-radius: 4px; }
        h2 { color: #666; border-bottom: 2px solid #4CAF50; padding-bottom: 10px; }
    </style>
</head>
<body>
    <div class="container">
        <h1>🔍 Feeding System Diagnostic</h1>
        
        <?php
        try {
            $pdo = getDBConnection();
            echo "<div class='success'>✅ Database connected</div>";
            
            // Check tables
            echo "<h2>Database Tables</h2>";
            $tables = $pdo->query("SHOW TABLES")->fetchAll(PDO::FETCH_COLUMN);
            
            $requiredTables = ['class_feeding_fees', 'feeding_records', 'students'];
            foreach ($requiredTables as $table) {
                if (in_array($table, $tables)) {
                    echo "<div class='success'>✅ $table exists</div>";
                    
                    // Show structure
                    $columns = $pdo->query("SHOW COLUMNS FROM $table")->fetchAll(PDO::FETCH_COLUMN);
                    echo "<div class='info'>Columns: " . implode(', ', $columns) . "</div>";
                    
                    // Show count
                    $count = $pdo->query("SELECT COUNT(*) FROM $table")->fetchColumn();
                    echo "<div class='info'>Records: $count</div>";
                } else {
                    echo "<div class='error'>❌ $table MISSING</div>";
                }
            }
            
            // Check if class_feeding_fees has data
            if (in_array('class_feeding_fees', $tables)) {
                echo "<h2>Class Feeding Fees</h2>";
                $fees = $pdo->query("SELECT * FROM class_feeding_fees")->fetchAll(PDO::FETCH_ASSOC);
                if (empty($fees)) {
                    echo "<div class='error'>❌ No feeding fees configured</div>";
                    echo "<div class='info'>Run: /school/database/create_feeding_tables.php</div>";
                } else {
                    echo "<div class='success'>✅ " . count($fees) . " classes configured</div>";
                    foreach ($fees as $fee) {
                        echo "<div class='info'>{$fee['class_name']}: GH₵{$fee['daily_fee']}</div>";
                    }
                }
            }
            
            // Check students table columns
            if (in_array('students', $tables)) {
                echo "<h2>Students Table - Feeding Columns</h2>";
                $columns = $pdo->query("SHOW COLUMNS FROM students")->fetchAll(PDO::FETCH_COLUMN);
                $feedingCols = [
                    'feeding_balance',
                    'feeding_weekly_balance', 
                    'feeding_payment_frequency',
                    'custom_feeding_fee',
                    'feeding_scholarship_percentage',
                    'feeding_notes'
                ];
                
                foreach ($feedingCols as $col) {
                    if (in_array($col, $columns)) {
                        echo "<div class='success'>✅ $col</div>";
                    } else {
                        echo "<div class='error'>❌ $col MISSING</div>";
                    }
                }
            }
            
            // Test API endpoint
            echo "<h2>API Test</h2>";
            echo "<div class='info'>Try calling: /school/api/feeding-fees</div>";
            
            // Summary
            echo "<h2>Summary</h2>";
            $allGood = in_array('class_feeding_fees', $tables) && 
                       in_array('feeding_records', $tables) &&
                       !empty($pdo->query("SELECT COUNT(*) FROM class_feeding_fees")->fetchColumn());
            
            if ($allGood) {
                echo "<div class='success' style='font-size: 18px; padding: 20px;'>
                    ✅ <strong>Everything looks good!</strong><br>
                    Your feeding system should work.
                </div>";
            } else {
                echo "<div class='error' style='font-size: 18px; padding: 20px;'>
                    ❌ <strong>Issues found!</strong><br>
                    Please run: <a href='../database/create_feeding_tables.php'>/school/database/create_feeding_tables.php</a>
                </div>";
            }
            
        } catch (Exception $e) {
            echo "<div class='error'>❌ Error: " . htmlspecialchars($e->getMessage()) . "</div>";
        }
        ?>
    </div>
</body>
</html>
