<?php
/**
 * SMS Handler for Mnotify Integration
 * Handles payment reminders, receipts, announcements, and SMS history
 */

/**
 * Handle SMS requests
 */
function handleSMS($conn, $method) {
    $user = requireAuth(); // Require authentication for all SMS endpoints
    
    switch($method) {
        case 'GET':
            getSMSHistory($conn);
            break;
            
        case 'POST':
            $data = json_decode(file_get_contents('php://input'), true);
            $action = $data['action'] ?? '';
            
            switch($action) {
                case 'send-payment-reminder':
                    sendPaymentReminder($conn, $data);
                    break;
                    
                case 'send-receipt':
                    sendReceipt($conn, $data);
                    break;
                    
                case 'send-announcement':
                    sendAnnouncement($conn, $data);
                    break;
                    
                case 'send-bulk-reminders':
                    sendBulkPaymentReminders($conn, $data);
                    break;
                    
                default:
                    http_response_code(400);
                    echo json_encode([
                        'success' => false,
                        'error' => 'Invalid action. Valid actions: send-payment-reminder, send-receipt, send-announcement, send-bulk-reminders'
                    ]);
                    break;
            }
            break;
            
        default:
            http_response_code(405);
            echo json_encode(['success' => false, 'error' => 'Method not allowed']);
            break;
    }
}

/**
 * Send SMS via Mnotify API
 */
function sendSMS($recipient, $message) {
    $apiKey = defined('SMS_API_KEY') ? SMS_API_KEY : 'J2rI5TXxDosrbwlmNyoPbwtK6';
    $senderId = defined('SMS_SENDER_ID') ? SMS_SENDER_ID : 'SoNice';
    $apiUrl = defined('SMS_API_URL') ? SMS_API_URL : 'https://api.mnotify.com/api/sms/quick';
    
    // Remove any non-numeric characters from phone number (except +)
    $recipient = preg_replace('/[^0-9+]/', '', $recipient);
    
    // Ensure phone number starts with country code (Ghana: 233)
    if (!preg_match('/^\+?233/', $recipient)) {
        // If it starts with 0, replace with 233
        if (preg_match('/^0/', $recipient)) {
            $recipient = '233' . substr($recipient, 1);
        } elseif (!preg_match('/^\+?233/', $recipient)) {
            $recipient = '233' . $recipient;
        }
    }
    
    // Remove + if present
    $recipient = str_replace('+', '', $recipient);
    
    $postData = [
        'key' => $apiKey,
        'to' => $recipient,
        'msg' => $message,
        'sender_id' => $senderId
    ];
    
    // Initialize cURL
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $apiUrl);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($postData));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    curl_close($ch);
    
    if ($curlError) {
        return [
            'success' => false,
            'error' => 'cURL Error: ' . $curlError
        ];
    }
    
    $responseData = json_decode($response, true);
    
    if ($httpCode === 200 && isset($responseData['status']) && $responseData['status'] === 'success') {
        return [
            'success' => true,
            'message' => 'SMS sent successfully',
            'response' => $responseData
        ];
    } else {
        return [
            'success' => false,
            'error' => $responseData['message'] ?? 'Failed to send SMS',
            'response' => $responseData,
            'http_code' => $httpCode
        ];
    }
}

/**
 * Log SMS to database
 */
function logSMS($conn, $recipient, $message, $type, $studentId = null, $metadata = null) {
    try {
        // Create sms_logs table if it doesn't exist
        $conn->exec("
            CREATE TABLE IF NOT EXISTS sms_logs (
                id INT AUTO_INCREMENT PRIMARY KEY,
                recipient VARCHAR(20) NOT NULL,
                message TEXT NOT NULL,
                type VARCHAR(50) NOT NULL,
                student_id VARCHAR(50),
                metadata TEXT,
                status VARCHAR(20) DEFAULT 'sent',
                sent_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                INDEX idx_recipient (recipient),
                INDEX idx_type (type),
                INDEX idx_sent_at (sent_at)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4
        ");
        
        $stmt = $conn->prepare("
            INSERT INTO sms_logs (recipient, message, type, student_id, metadata, status)
            VALUES (?, ?, ?, ?, ?, 'sent')
        ");
        
        $metadataJson = $metadata ? json_encode($metadata) : null;
        $stmt->execute([$recipient, $message, $type, $studentId, $metadataJson]);
        
        return $conn->lastInsertId();
    } catch (PDOException $e) {
        error_log("Error logging SMS: " . $e->getMessage());
        return false;
    }
}

/**
 * Send payment reminder SMS
 */
function sendPaymentReminder($conn, $data) {
    try {
        $studentId = $data['studentId'] ?? null;
        $term = $data['term'] ?? 'Term 1';
        
        if (!$studentId) {
            http_response_code(400);
            echo json_encode(['success' => false, 'error' => 'Student ID is required']);
            return;
        }
        
        // Get student information
        $stmt = $conn->prepare("
            SELECT 
                s.student_id,
                CONCAT(s.first_name, ' ', s.last_name) as student_name,
                s.class,
                s.phone_number as parent_phone,
                s.father_name as parent_name,
                COALESCE(SUM(f.amount - f.amount_paid - COALESCE(f.discount, 0)), 0) as outstanding_balance
            FROM students s
            LEFT JOIN fees f ON f.student_id = s.id AND f.status != 'Paid'
            WHERE s.student_id = ? OR s.id = ?
            GROUP BY s.id
        ");
        $stmt->execute([$studentId, $studentId]);
        $student = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$student) {
            http_response_code(404);
            echo json_encode(['success' => false, 'error' => 'Student not found']);
            return;
        }
        
        if (empty($student['parent_phone'])) {
            http_response_code(400);
            echo json_encode(['success' => false, 'error' => 'Parent phone number not available for this student']);
            return;
        }
        
        // Format outstanding balance
        $balance = floatval($student['outstanding_balance']);
        $balanceFormatted = number_format($balance, 2);
        
        // Create message
        $parentName = $student['parent_name'] ? $student['parent_name'] : 'Parent';
        $message = "Hello {$parentName}, this is Divine Grace School. Your child {$student['student_name']} ({$student['student_id']}) in {$student['class']} has an outstanding balance of GH₵{$balanceFormatted} for {$term}. Please visit the school to make payment. Thank you.";
        
        // Send SMS
        $smsResult = sendSMS($student['parent_phone'], $message);
        
        if ($smsResult['success']) {
            // Log SMS
            logSMS($conn, $student['parent_phone'], $message, 'payment_reminder', $student['student_id'], [
                'term' => $term,
                'outstanding_balance' => $balance
            ]);
            
            echo json_encode([
                'success' => true,
                'message' => 'Payment reminder sent successfully',
                'recipient' => $student['parent_phone'],
                'student' => $student['student_name']
            ]);
        } else {
            http_response_code(500);
            echo json_encode([
                'success' => false,
                'error' => $smsResult['error'],
                'recipient' => $student['parent_phone']
            ]);
        }
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['success' => false, 'error' => 'Database error: ' . $e->getMessage()]);
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    }
}

/**
 * Send receipt SMS
 */
function sendReceipt($conn, $data) {
    try {
        $studentId = $data['studentId'] ?? null;
        $amount = $data['amount'] ?? null;
        $paymentDate = $data['paymentDate'] ?? date('Y-m-d');
        $receiptNumber = $data['receiptNumber'] ?? null;
        
        if (!$studentId || !$amount) {
            http_response_code(400);
            echo json_encode(['success' => false, 'error' => 'Student ID and amount are required']);
            return;
        }
        
        // Get student information
        $stmt = $conn->prepare("
            SELECT 
                s.student_id,
                CONCAT(s.first_name, ' ', s.last_name) as student_name,
                s.class,
                s.phone_number as parent_phone,
                s.father_name as parent_name
            FROM students s
            WHERE s.student_id = ? OR s.id = ?
        ");
        $stmt->execute([$studentId, $studentId]);
        $student = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$student) {
            http_response_code(404);
            echo json_encode(['success' => false, 'error' => 'Student not found']);
            return;
        }
        
        if (empty($student['parent_phone'])) {
            http_response_code(400);
            echo json_encode(['success' => false, 'error' => 'Parent phone number not available for this student']);
            return;
        }
        
        // Format amount
        $amountFormatted = number_format(floatval($amount), 2);
        $receiptInfo = $receiptNumber ? "Receipt #{$receiptNumber}. " : '';
        
        // Create message
        $parentName = $student['parent_name'] ? $student['parent_name'] : 'Parent';
        $message = "Hello {$parentName}, thank you for your payment. {$receiptInfo}Amount: GH₵{$amountFormatted} for {$student['student_name']} ({$student['student_id']}) on {$paymentDate}. Divine Grace School.";
        
        // Send SMS
        $smsResult = sendSMS($student['parent_phone'], $message);
        
        if ($smsResult['success']) {
            // Log SMS
            logSMS($conn, $student['parent_phone'], $message, 'receipt', $student['student_id'], [
                'amount' => $amount,
                'payment_date' => $paymentDate,
                'receipt_number' => $receiptNumber
            ]);
            
            echo json_encode([
                'success' => true,
                'message' => 'Receipt SMS sent successfully',
                'recipient' => $student['parent_phone'],
                'student' => $student['student_name']
            ]);
        } else {
            http_response_code(500);
            echo json_encode([
                'success' => false,
                'error' => $smsResult['error'],
                'recipient' => $student['parent_phone']
            ]);
        }
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['success' => false, 'error' => 'Database error: ' . $e->getMessage()]);
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    }
}

/**
 * Send announcement SMS
 */
function sendAnnouncement($conn, $data) {
    try {
        $message = $data['message'] ?? '';
        $classFilter = $data['classFilter'] ?? null;
        $recipients = $data['recipients'] ?? [];
        
        if (empty($message)) {
            http_response_code(400);
            echo json_encode(['success' => false, 'error' => 'Message is required']);
            return;
        }
        
        // Get recipients
        $phones = [];
        if (!empty($recipients) && is_array($recipients)) {
            // Use provided recipients list
            $phones = array_filter($recipients);
        } else {
            // Get all parents or filtered by class
            $sql = "SELECT DISTINCT phone_number FROM students WHERE status = 'active' AND phone_number IS NOT NULL AND phone_number != ''";
            $params = [];
            
            if ($classFilter) {
                $sql .= " AND class = ?";
                $params[] = $classFilter;
            }
            
            $stmt = $conn->prepare($sql);
            $stmt->execute($params);
            $results = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            foreach ($results as $row) {
                if (!empty($row['phone_number'])) {
                    $phones[] = $row['phone_number'];
                }
            }
        }
        
        if (empty($phones)) {
            http_response_code(400);
            echo json_encode(['success' => false, 'error' => 'No recipients found']);
            return;
        }
        
        // Send SMS to all recipients
        $successCount = 0;
        $failureCount = 0;
        $failures = [];
        
        foreach ($phones as $phone) {
            $smsResult = sendSMS($phone, $message);
            
            if ($smsResult['success']) {
                $successCount++;
                logSMS($conn, $phone, $message, 'announcement', null, [
                    'class_filter' => $classFilter
                ]);
            } else {
                $failureCount++;
                $failures[] = ['phone' => $phone, 'error' => $smsResult['error']];
            }
            
            // Small delay to avoid rate limiting
            usleep(100000); // 0.1 second
        }
        
        echo json_encode([
            'success' => $failureCount === 0,
            'message' => "Announcement sent to {$successCount} recipient(s)",
            'sent' => $successCount,
            'failed' => $failureCount,
            'failures' => $failures
        ]);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['success' => false, 'error' => 'Database error: ' . $e->getMessage()]);
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    }
}

/**
 * Send bulk payment reminders
 */
function sendBulkPaymentReminders($conn, $data) {
    try {
        $term = $data['term'] ?? 'Term 1';
        $classFilter = $data['classFilter'] ?? null;
        
        // Get all students with outstanding balances
        $sql = "
            SELECT 
                s.student_id,
                s.id as student_db_id,
                CONCAT(s.first_name, ' ', s.last_name) as student_name,
                s.class,
                s.phone_number as parent_phone,
                s.father_name as parent_name,
                COALESCE(SUM(f.amount - f.amount_paid - COALESCE(f.discount, 0)), 0) as outstanding_balance
            FROM students s
            LEFT JOIN fees f ON f.student_id = s.id AND f.status != 'Paid'
            WHERE s.status = 'active' 
            AND s.phone_number IS NOT NULL 
            AND s.phone_number != ''
        ";
        
        $params = [];
        if ($classFilter) {
            $sql .= " AND s.class = ?";
            $params[] = $classFilter;
        }
        
        $sql .= " GROUP BY s.id HAVING outstanding_balance > 0";
        
        $stmt = $conn->prepare($sql);
        $stmt->execute($params);
        $students = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        if (empty($students)) {
            echo json_encode([
                'success' => true,
                'message' => 'No students with outstanding balances found',
                'sent' => 0,
                'failed' => 0
            ]);
            return;
        }
        
        $successCount = 0;
        $failureCount = 0;
        $failures = [];
        
        foreach ($students as $student) {
            $balance = floatval($student['outstanding_balance']);
            $balanceFormatted = number_format($balance, 2);
            
            $parentName = $student['parent_name'] ? $student['parent_name'] : 'Parent';
            $message = "Hello {$parentName}, this is Divine Grace School. Your child {$student['student_name']} ({$student['student_id']}) in {$student['class']} has an outstanding balance of GH₵{$balanceFormatted} for {$term}. Please visit the school to make payment. Thank you.";
            
            $smsResult = sendSMS($student['parent_phone'], $message);
            
            if ($smsResult['success']) {
                $successCount++;
                logSMS($conn, $student['parent_phone'], $message, 'bulk_payment_reminder', $student['student_id'], [
                    'term' => $term,
                    'outstanding_balance' => $balance
                ]);
            } else {
                $failureCount++;
                $failures[] = [
                    'student' => $student['student_name'],
                    'phone' => $student['parent_phone'],
                    'error' => $smsResult['error']
                ];
            }
            
            // Small delay to avoid rate limiting
            usleep(200000); // 0.2 second
        }
        
        echo json_encode([
            'success' => $failureCount === 0,
            'message' => "Bulk payment reminders sent to {$successCount} parent(s)",
            'sent' => $successCount,
            'failed' => $failureCount,
            'failures' => $failures
        ]);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['success' => false, 'error' => 'Database error: ' . $e->getMessage()]);
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    }
}

/**
 * Get SMS history
 */
function getSMSHistory($conn) {
    try {
        $limit = isset($_GET['limit']) ? intval($_GET['limit']) : 50;
        $offset = isset($_GET['offset']) ? intval($_GET['offset']) : 0;
        $type = $_GET['type'] ?? null;
        
        // Create sms_logs table if it doesn't exist
        $conn->exec("
            CREATE TABLE IF NOT EXISTS sms_logs (
                id INT AUTO_INCREMENT PRIMARY KEY,
                recipient VARCHAR(20) NOT NULL,
                message TEXT NOT NULL,
                type VARCHAR(50) NOT NULL,
                student_id VARCHAR(50),
                metadata TEXT,
                status VARCHAR(20) DEFAULT 'sent',
                sent_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                INDEX idx_recipient (recipient),
                INDEX idx_type (type),
                INDEX idx_sent_at (sent_at)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4
        ");
        
        $sql = "SELECT * FROM sms_logs WHERE 1=1";
        $params = [];
        
        if ($type) {
            $sql .= " AND type = ?";
            $params[] = $type;
        }
        
        $sql .= " ORDER BY sent_at DESC LIMIT ? OFFSET ?";
        $params[] = $limit;
        $params[] = $offset;
        
        $stmt = $conn->prepare($sql);
        $stmt->execute($params);
        $logs = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Get total count
        $countSql = "SELECT COUNT(*) as total FROM sms_logs WHERE 1=1";
        $countParams = [];
        if ($type) {
            $countSql .= " AND type = ?";
            $countParams[] = $type;
        }
        $countStmt = $conn->prepare($countSql);
        $countStmt->execute($countParams);
        $total = $countStmt->fetch(PDO::FETCH_ASSOC)['total'];
        
        // Parse metadata JSON
        foreach ($logs as &$log) {
            if ($log['metadata']) {
                $log['metadata'] = json_decode($log['metadata'], true);
            }
        }
        
        echo json_encode([
            'success' => true,
            'data' => $logs,
            'pagination' => [
                'total' => (int)$total,
                'limit' => $limit,
                'offset' => $offset
            ]
        ]);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['success' => false, 'error' => 'Database error: ' . $e->getMessage()]);
    }
}
?>

