<?php
/**
 * Extended Assessment System API Handler
 * Handles class-based subject management and student assessments
 */

require_once __DIR__ . '/../../config/database.php';

function handleAssessmentsExtended($method, $endpoint, $data = null) {
    try {
        $conn = getDBConnection();
        
        switch ($endpoint) {
            // CLASS SUBJECTS (Extended)
            case 'class-subjects-extended':
                return handleClassSubjectsExtended($conn, $method, $data);
            case 'class-subjects-extended/by-class':
                return getClassSubjectsExtended($conn, $data['class_id'] ?? null);
            
            // STUDENT ASSESSMENTS
            case 'student-assessments':
                return handleStudentAssessments($conn, $method, $data);
            case 'student-assessments/by-student':
                return getAssessmentsByStudent($conn, $data);
            case 'student-assessments/by-class':
                return getAssessmentsByClassExtended($conn, $data);
            
            // REPORT CARDS (Extended)
            case 'report-cards-extended':
                return handleReportCardsExtended($conn, $method, $data);
            case 'report-cards-extended/generate':
                return generateReportCardExtended($conn, $data);
            case 'report-cards-extended/by-student':
                return getReportCardExtended($conn, $data);
            
            // UTILITIES
            case 'calculate-grade':
                return calculateGrade($conn, $data['score'] ?? 0);
            case 'class-positions':
                return calculateClassPositions($conn, $data);
            
            default:
                return ['success' => false, 'message' => "Invalid endpoint: $endpoint"];
        }
    } catch (Exception $e) {
        error_log("Assessments Extended Handler Error: " . $e->getMessage());
        return ['success' => false, 'message' => $e->getMessage()];
    }
}

// ============================================
// CLASS SUBJECTS (EXTENDED)
// ============================================

function handleClassSubjectsExtended($conn, $method, $data) {
    switch ($method) {
        case 'GET':
            return getClassSubjectsExtended($conn, $data['class_id'] ?? null);
        case 'POST':
            return addClassSubjectExtended($conn, $data);
        case 'PUT':
            return updateClassSubjectExtended($conn, $data);
        case 'DELETE':
            return deleteClassSubjectExtended($conn, $data['id'] ?? null);
        default:
            return ['success' => false, 'message' => 'Invalid method'];
    }
}

function getClassSubjectsExtended($conn, $classId) {
    if (!$classId) {
        return ['success' => false, 'message' => 'Class ID is required'];
    }
    
    $sql = "SELECT cse.*, c.class_name, u.username as teacher_name
            FROM class_subjects_extended cse
            JOIN classes c ON cse.class_id = c.id
            LEFT JOIN users u ON cse.teacher_id = u.id
            WHERE cse.class_id = ? AND cse.is_active = 1
            ORDER BY cse.subject_name";
    
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("i", $classId);
    $stmt->execute();
    $result = $stmt->get_result();
    
    $subjects = [];
    while ($row = $result->fetch_assoc()) {
        $subjects[] = $row;
    }
    
    return ['success' => true, 'data' => $subjects];
}

function addClassSubjectExtended($conn, $data) {
    $stmt = $conn->prepare("INSERT INTO class_subjects_extended (class_id, subject_name, subject_code, teacher_id, max_class_score, max_exam_score) VALUES (?, ?, ?, ?, ?, ?)");
    
    $maxClass = $data['max_class_score'] ?? 40.00;
    $maxExam = $data['max_exam_score'] ?? 60.00;
    $teacherId = $data['teacher_id'] ?? null;
    
    $stmt->bind_param("issiii", 
        $data['class_id'], 
        $data['subject_name'], 
        $data['subject_code'], 
        $teacherId,
        $maxClass,
        $maxExam
    );
    
    if ($stmt->execute()) {
        return ['success' => true, 'message' => 'Subject added successfully', 'id' => $conn->insert_id];
    }
    return ['success' => false, 'message' => 'Failed to add subject: ' . $conn->error];
}

function updateClassSubjectExtended($conn, $data) {
    $stmt = $conn->prepare("UPDATE class_subjects_extended SET subject_name = ?, subject_code = ?, teacher_id = ?, max_class_score = ?, max_exam_score = ? WHERE id = ?");
    
    $stmt->bind_param("ssiidi", 
        $data['subject_name'], 
        $data['subject_code'], 
        $data['teacher_id'],
        $data['max_class_score'],
        $data['max_exam_score'],
        $data['id']
    );
    
    if ($stmt->execute()) {
        return ['success' => true, 'message' => 'Subject updated successfully'];
    }
    return ['success' => false, 'message' => 'Failed to update subject'];
}

function deleteClassSubjectExtended($conn, $id) {
    $stmt = $conn->prepare("UPDATE class_subjects_extended SET is_active = 0 WHERE id = ?");
    $stmt->bind_param("i", $id);
    
    if ($stmt->execute()) {
        return ['success' => true, 'message' => 'Subject deleted successfully'];
    }
    return ['success' => false, 'message' => 'Failed to delete subject'];
}

// ============================================
// STUDENT ASSESSMENTS
// ============================================

function handleStudentAssessments($conn, $method, $data) {
    switch ($method) {
        case 'POST':
            return saveStudentAssessment($conn, $data);
        case 'PUT':
            return updateStudentAssessment($conn, $data);
        case 'DELETE':
            return deleteStudentAssessment($conn, $data['id'] ?? null);
        default:
            return ['success' => false, 'message' => 'Invalid method'];
    }
}

function getAssessmentsByStudent($conn, $data) {
    $studentId = $data['student_id'] ?? null;
    $termId = $data['term_id'] ?? null;
    
    if (!$studentId) {
        return ['success' => false, 'message' => 'Student ID is required'];
    }
    
    $sql = "SELECT sa.*, cse.subject_name, cse.subject_code, cse.max_class_score, cse.max_exam_score,
            t.term_name, t.academic_year
            FROM student_assessments sa
            JOIN class_subjects_extended cse ON sa.class_subject_id = cse.id
            JOIN academic_terms t ON sa.term_id = t.id
            WHERE sa.student_id = ?";
    
    if ($termId) {
        $sql .= " AND sa.term_id = ?";
        $stmt = $conn->prepare($sql);
        $stmt->bind_param("ii", $studentId, $termId);
    } else {
        $stmt = $conn->prepare($sql);
        $stmt->bind_param("i", $studentId);
    }
    
    $stmt->execute();
    $result = $stmt->get_result();
    
    $assessments = [];
    while ($row = $result->fetch_assoc()) {
        $assessments[] = $row;
    }
    
    return ['success' => true, 'data' => $assessments];
}

function getAssessmentsByClassExtended($conn, $data) {
    $classId = $data['class_id'] ?? null;
    $termId = $data['term_id'] ?? null;
    
    if (!$classId || !$termId) {
        return ['success' => false, 'message' => 'Class ID and Term ID are required'];
    }
    
    // Get all students in the class with their assessments
    $sql = "SELECT s.id as student_id, s.student_id as student_number,
            CONCAT(s.first_name, ' ', s.last_name) as student_name,
            cse.id as class_subject_id, cse.subject_name,
            sa.id as assessment_id, sa.class_score, sa.exam_score, sa.total_score, sa.grade, sa.teacher_remark
            FROM students s
            CROSS JOIN class_subjects_extended cse
            LEFT JOIN student_assessments sa ON s.id = sa.student_id 
                AND cse.id = sa.class_subject_id 
                AND sa.term_id = ?
            WHERE cse.class_id = ? AND cse.is_active = 1
            AND (s.class_id = ? OR s.class = (SELECT class_name FROM classes WHERE id = ?))
            AND s.status = 'active'
            ORDER BY student_name, cse.subject_name";
    
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("iiii", $termId, $classId, $classId, $classId);
    $stmt->execute();
    $result = $stmt->get_result();
    
    $assessments = [];
    while ($row = $result->fetch_assoc()) {
        $assessments[] = $row;
    }
    
    return ['success' => true, 'data' => $assessments];
}

function saveStudentAssessment($conn, $data) {
    // Validate scores
    $classScore = $data['class_score'] ?? 0;
    $examScore = $data['exam_score'] ?? 0;
    
    if ($classScore > 50) {
        return ['success' => false, 'message' => 'Class score cannot exceed 50'];
    }
    
    if ($examScore > 50) {
        return ['success' => false, 'message' => 'Exam score cannot exceed 50'];
    }
    
    $totalScore = $classScore + $examScore;
    
    // Calculate grade and remark
    $gradeData = calculateGradeAndRemark($conn, $totalScore);
    
    $stmt = $conn->prepare("INSERT INTO student_assessments 
        (student_id, class_subject_id, term_id, class_score, exam_score, grade, teacher_remark, recorded_by) 
        VALUES (?, ?, ?, ?, ?, ?, ?, ?)
        ON DUPLICATE KEY UPDATE 
        class_score = ?, exam_score = ?, grade = ?, teacher_remark = ?, updated_at = CURRENT_TIMESTAMP");
    
    $stmt->bind_param("iiiidssissds", 
        $data['student_id'],
        $data['class_subject_id'],
        $data['term_id'],
        $classScore,
        $examScore,
        $gradeData['grade'],
        $gradeData['remark'],
        $data['recorded_by'],
        $classScore,
        $examScore,
        $gradeData['grade'],
        $gradeData['remark']
    );
    
    if ($stmt->execute()) {
        return [
            'success' => true, 
            'message' => 'Assessment saved successfully',
            'data' => [
                'total_score' => $totalScore,
                'grade' => $gradeData['grade'],
                'remark' => $gradeData['remark']
            ]
        ];
    }
    return ['success' => false, 'message' => 'Failed to save assessment: ' . $conn->error];
}

function updateStudentAssessment($conn, $data) {
    return saveStudentAssessment($conn, $data);
}

function deleteStudentAssessment($conn, $id) {
    $stmt = $conn->prepare("DELETE FROM student_assessments WHERE id = ?");
    $stmt->bind_param("i", $id);
    
    if ($stmt->execute()) {
        return ['success' => true, 'message' => 'Assessment deleted successfully'];
    }
    return ['success' => false, 'message' => 'Failed to delete assessment'];
}

// ============================================
// REPORT CARDS (EXTENDED)
// ============================================

function handleReportCardsExtended($conn, $method, $data) {
    switch ($method) {
        case 'GET':
            return getReportCardExtended($conn, $data);
        case 'POST':
            return generateReportCardExtended($conn, $data);
        case 'PUT':
            return updateReportCardCommentsExtended($conn, $data);
        default:
            return ['success' => false, 'message' => 'Invalid method'];
    }
}

function getReportCardExtended($conn, $data) {
    $studentId = $data['student_id'] ?? null;
    $termId = $data['term_id'] ?? null;
    
    if (!$studentId || !$termId) {
        return ['success' => false, 'message' => 'Student ID and Term ID are required'];
    }
    
    // Get student info
    $stmt = $conn->prepare("SELECT s.*, c.class_name 
        FROM students s 
        LEFT JOIN classes c ON s.class_id = c.id OR s.class = c.class_name
        WHERE s.id = ?");
    $stmt->bind_param("i", $studentId);
    $stmt->execute();
    $student = $stmt->get_result()->fetch_assoc();
    
    if (!$student) {
        return ['success' => false, 'message' => 'Student not found'];
    }
    
    // Get term info
    $stmt = $conn->prepare("SELECT * FROM academic_terms WHERE id = ?");
    $stmt->bind_param("i", $termId);
    $stmt->execute();
    $term = $stmt->get_result()->fetch_assoc();
    
    // Get all assessments for this student and term
    $sql = "SELECT sa.*, cse.subject_name, cse.subject_code
            FROM student_assessments sa
            JOIN class_subjects_extended cse ON sa.class_subject_id = cse.id
            WHERE sa.student_id = ? AND sa.term_id = ?
            ORDER BY cse.subject_name";
    
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("ii", $studentId, $termId);
    $stmt->execute();
    $result = $stmt->get_result();
    
    $subjects = [];
    $totalCWA = 0;
    $subjectCount = 0;
    
    while ($row = $result->fetch_assoc()) {
        $subjects[] = $row;
        $totalCWA += $row['total_score'];
        $subjectCount++;
    }
    
    $averageScore = $subjectCount > 0 ? $totalCWA / $subjectCount : 0;
    
    // Calculate class position
    $classId = $student['class_id'] ?? null;
    if (!$classId) {
        // Get class_id from class name
        $stmt = $conn->prepare("SELECT id FROM classes WHERE class_name = ?");
        $stmt->bind_param("s", $student['class']);
        $stmt->execute();
        $classResult = $stmt->get_result()->fetch_assoc();
        $classId = $classResult['id'] ?? null;
    }
    
    $position = calculateStudentPosition($conn, $studentId, $termId, $classId, $averageScore);
    
    // Get report card comments if exists
    $stmt = $conn->prepare("SELECT teacher_comment, headteacher_comment, attendance_percentage, days_present, days_absent 
        FROM report_cards_extended 
        WHERE student_id = ? AND term_id = ?");
    $stmt->bind_param("ii", $studentId, $termId);
    $stmt->execute();
    $reportData = $stmt->get_result()->fetch_assoc();
    
    return [
        'success' => true,
        'data' => [
            'student' => $student,
            'term' => $term,
            'subjects' => $subjects,
            'total_cwa' => $totalCWA,
            'average_score' => round($averageScore, 2),
            'class_position' => $position['position'],
            'total_students' => $position['total_students'],
            'teacher_comment' => $reportData['teacher_comment'] ?? '',
            'headteacher_comment' => $reportData['headteacher_comment'] ?? '',
            'attendance_percentage' => $reportData['attendance_percentage'] ?? 0,
            'days_present' => $reportData['days_present'] ?? 0,
            'days_absent' => $reportData['days_absent'] ?? 0
        ]
    ];
}

function generateReportCardExtended($conn, $data) {
    $studentId = $data['student_id'] ?? null;
    $termId = $data['term_id'] ?? null;
    $generatedBy = $data['generated_by'] ?? null;
    
    if (!$studentId || !$termId || !$generatedBy) {
        return ['success' => false, 'message' => 'Missing required fields'];
    }
    
    // Get student's class
    $stmt = $conn->prepare("SELECT class_id, class FROM students WHERE id = ?");
    $stmt->bind_param("i", $studentId);
    $stmt->execute();
    $student = $stmt->get_result()->fetch_assoc();
    
    $classId = $student['class_id'];
    if (!$classId) {
        $stmt = $conn->prepare("SELECT id FROM classes WHERE class_name = ?");
        $stmt->bind_param("s", $student['class']);
        $stmt->execute();
        $classResult = $stmt->get_result()->fetch_assoc();
        $classId = $classResult['id'] ?? null;
    }
    
    // Calculate totals
    $sql = "SELECT SUM(total_score) as total_cwa, AVG(total_score) as average_score, COUNT(*) as subject_count
            FROM student_assessments
            WHERE student_id = ? AND term_id = ?";
    
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("ii", $studentId, $termId);
    $stmt->execute();
    $totals = $stmt->get_result()->fetch_assoc();
    
    $totalCWA = $totals['total_cwa'] ?? 0;
    $averageScore = $totals['average_score'] ?? 0;
    
    // Calculate position
    $position = calculateStudentPosition($conn, $studentId, $termId, $classId, $averageScore);
    
    // Get attendance summary
    $attendance = getAttendanceSummaryForTermExtended($conn, $studentId, $termId);
    
    // Save report card
    $stmt = $conn->prepare("INSERT INTO report_cards_extended 
        (student_id, term_id, class_id, total_cwa, average_score, class_position, total_students, 
         attendance_percentage, days_present, days_absent, generated_by)
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        ON DUPLICATE KEY UPDATE
        total_cwa = ?, average_score = ?, class_position = ?, total_students = ?,
        attendance_percentage = ?, days_present = ?, days_absent = ?, generated_at = CURRENT_TIMESTAMP");
    
    $stmt->bind_param("iiiidiidiiiidiidii",
        $studentId, $termId, $classId, $totalCWA, $averageScore, 
        $position['position'], $position['total_students'],
        $attendance['percentage'], $attendance['present'], $attendance['absent'],
        $generatedBy,
        $totalCWA, $averageScore, $position['position'], $position['total_students'],
        $attendance['percentage'], $attendance['present'], $attendance['absent']
    );
    
    if ($stmt->execute()) {
        return ['success' => true, 'message' => 'Report card generated successfully'];
    }
    return ['success' => false, 'message' => 'Failed to generate report card'];
}

function updateReportCardCommentsExtended($conn, $data) {
    $stmt = $conn->prepare("UPDATE report_cards_extended 
        SET teacher_comment = ?, headteacher_comment = ?, is_published = ?
        WHERE student_id = ? AND term_id = ?");
    
    $isPublished = $data['is_published'] ?? 0;
    $stmt->bind_param("ssiii", 
        $data['teacher_comment'],
        $data['headteacher_comment'],
        $isPublished,
        $data['student_id'],
        $data['term_id']
    );
    
    if ($stmt->execute()) {
        if ($isPublished) {
            $stmt = $conn->prepare("UPDATE report_cards_extended SET published_at = NOW() 
                WHERE student_id = ? AND term_id = ?");
            $stmt->bind_param("ii", $data['student_id'], $data['term_id']);
            $stmt->execute();
        }
        return ['success' => true, 'message' => 'Report card updated successfully'];
    }
    return ['success' => false, 'message' => 'Failed to update report card'];
}

// ============================================
// HELPER FUNCTIONS
// ============================================

function calculateGrade($conn, $score) {
    return calculateGradeAndRemark($conn, $score);
}

function calculateGradeAndRemark($conn, $score) {
    $stmt = $conn->prepare("SELECT gs.grade, tr.remark 
        FROM grading_scale gs
        LEFT JOIN teacher_remarks tr ON gs.grade = tr.grade
        WHERE ? BETWEEN gs.min_score AND gs.max_score AND gs.is_active = 1 
        LIMIT 1");
    $stmt->bind_param("d", $score);
    $stmt->execute();
    $result = $stmt->get_result()->fetch_assoc();
    
    return [
        'grade' => $result['grade'] ?? 'F',
        'remark' => $result['remark'] ?? 'No remark'
    ];
}

function calculateStudentPosition($conn, $studentId, $termId, $classId, $studentAverage) {
    // Get all students in the class with their averages
    $sql = "SELECT s.id, AVG(sa.total_score) as avg_score
            FROM students s
            JOIN student_assessments sa ON s.id = sa.student_id
            WHERE sa.term_id = ? 
            AND (s.class_id = ? OR s.class = (SELECT class_name FROM classes WHERE id = ?))
            AND s.status = 'active'
            GROUP BY s.id
            ORDER BY avg_score DESC";
    
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("iii", $termId, $classId, $classId);
    $stmt->execute();
    $result = $stmt->get_result();
    
    $position = 1;
    $totalStudents = 0;
    
    while ($row = $result->fetch_assoc()) {
        $totalStudents++;
        if ($row['id'] == $studentId) {
            break;
        }
        if ($row['avg_score'] > $studentAverage) {
            $position++;
        }
    }
    
    return [
        'position' => $position,
        'total_students' => $totalStudents
    ];
}

function calculateClassPositions($conn, $data) {
    $termId = $data['term_id'] ?? null;
    $classId = $data['class_id'] ?? null;
    
    if (!$termId || !$classId) {
        return ['success' => false, 'message' => 'Term ID and Class ID are required'];
    }
    
    $sql = "SELECT s.id, s.student_id, CONCAT(s.first_name, ' ', s.last_name) as name,
            AVG(sa.total_score) as average_score,
            SUM(sa.total_score) as total_cwa
            FROM students s
            JOIN student_assessments sa ON s.id = sa.student_id
            WHERE sa.term_id = ? 
            AND (s.class_id = ? OR s.class = (SELECT class_name FROM classes WHERE id = ?))
            AND s.status = 'active'
            GROUP BY s.id
            ORDER BY average_score DESC";
    
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("iii", $termId, $classId, $classId);
    $stmt->execute();
    $result = $stmt->get_result();
    
    $positions = [];
    $position = 1;
    
    while ($row = $result->fetch_assoc()) {
        $positions[] = [
            'student_id' => $row['id'],
            'student_number' => $row['student_id'],
            'name' => $row['name'],
            'average_score' => round($row['average_score'], 2),
            'total_cwa' => round($row['total_cwa'], 2),
            'position' => $position++
        ];
    }
    
    return ['success' => true, 'data' => $positions];
}

function getAttendanceSummaryForTermExtended($conn, $studentId, $termId) {
    $stmt = $conn->prepare("SELECT start_date, end_date FROM academic_terms WHERE id = ?");
    $stmt->bind_param("i", $termId);
    $stmt->execute();
    $term = $stmt->get_result()->fetch_assoc();
    
    if (!$term) {
        return ['percentage' => 0, 'present' => 0, 'absent' => 0];
    }
    
    $sql = "SELECT 
            COUNT(*) as total_days,
            SUM(CASE WHEN status = 'present' THEN 1 ELSE 0 END) as present_days,
            SUM(CASE WHEN status = 'absent' THEN 1 ELSE 0 END) as absent_days
            FROM attendance 
            WHERE student_id = ? AND attendance_date BETWEEN ? AND ?";
    
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("iss", $studentId, $term['start_date'], $term['end_date']);
    $stmt->execute();
    $summary = $stmt->get_result()->fetch_assoc();
    
    $percentage = $summary['total_days'] > 0 
        ? round(($summary['present_days'] / $summary['total_days']) * 100, 2) 
        : 0;
    
    return [
        'percentage' => $percentage,
        'present' => $summary['present_days'] ?? 0,
        'absent' => $summary['absent_days'] ?? 0
    ];
}
