<?php
/**
 * Academic System API Handler
 * Handles all academic-related operations
 */

// Don't set headers here - they're set in main index.php
require_once __DIR__ . '/../../config/database.php';

function handleAcademicRequest($method, $endpoint, $data = null) {
    try {
        // Check if database config is loaded
        if (!function_exists('getDBConnection')) {
            error_log("Academic Handler: getDBConnection function not found");
            return ['success' => false, 'message' => 'Database configuration error'];
        }
        
        $conn = getDBConnection();
        
        switch ($endpoint) {
            // CLASSES
            case 'classes':
                return handleClasses($conn, $method, $data);
            case 'classes/list':
                return getClasses($conn);
            case 'classes/students':
                return getClassStudents($conn, $data['class_id'] ?? null);
            
            // SUBJECTS
            case 'subjects':
                return handleSubjects($conn, $method, $data);
            case 'subjects/list':
                return getSubjects($conn);
            
            // CLASS-SUBJECTS
            case 'class-subjects':
                return handleClassSubjects($conn, $method, $data);
            case 'class-subjects/by-class':
                return getClassSubjects($conn, $data['class_id'] ?? null);
            
            // ACADEMIC TERMS
            case 'terms':
                return handleTerms($conn, $method, $data);
            case 'terms/current':
                return getCurrentTerm($conn);
            
            // ASSESSMENTS
            case 'assessments':
                return handleAssessments($conn, $method, $data);
            case 'assessments/by-class':
                return getAssessmentsByClass($conn, $data);
            
            // SCORES
            case 'scores':
                return handleAcademicScores($conn, $method, $data);
            case 'scores/by-assessment':
                return getScoresByAssessment($conn, $data['assessment_id'] ?? null);
            case 'scores/by-student':
                return getScoresByStudent($conn, $data['student_id'] ?? null, $data['term_id'] ?? null);
            
            // ATTENDANCE
            case 'attendance':
                return handleAttendance($conn, $method, $data);
            case 'attendance/by-class':
                return getAttendanceByClass($conn, $data);
            case 'attendance/by-student':
                return getAttendanceByStudent($conn, $data['student_id'] ?? null, $data);
            case 'attendance/summary':
                return getAttendanceSummary($conn, $data);
            
            // REPORT CARDS
            case 'report-cards':
                return handleReportCards($conn, $method, $data);
            case 'report-cards/generate':
                return generateReportCard($conn, $data);
            case 'report-cards/by-student':
                return getReportCardByStudent($conn, $data['student_id'] ?? null, $data['term_id'] ?? null);
            
            // GRADING SCALE
            case 'grading-scale':
                return getGradingScale($conn);
            
            // ASSESSMENT TYPES
            case 'assessment-types':
                return getAssessmentTypes($conn);
            
            default:
                error_log("Academic Handler: Invalid endpoint requested: $endpoint");
                return ['success' => false, 'message' => "Invalid endpoint: $endpoint"];
        }
    } catch (Exception $e) {
        error_log("Academic Handler Error: " . $e->getMessage());
        error_log("Stack trace: " . $e->getTraceAsString());
        return [
            'success' => false, 
            'message' => $e->getMessage(),
            'endpoint' => $endpoint ?? 'unknown'
        ];
    }
}

// ============================================
// CLASSES MANAGEMENT
// ============================================

function handleClasses($conn, $method, $data) {
    switch ($method) {
        case 'GET':
            return getClasses($conn);
        case 'POST':
            return createClass($conn, $data);
        case 'PUT':
            return updateClass($conn, $data);
        case 'DELETE':
            return deleteClass($conn, $data['id'] ?? null);
        default:
            return ['success' => false, 'message' => 'Invalid method'];
    }
}

function getClasses($conn) {
    $sql = "SELECT c.*, 
            (SELECT COUNT(*) FROM students WHERE (class = c.class_name OR class_id = c.id) AND status = 'active') as student_count
            FROM classes c 
            WHERE c.is_active = 1 
            ORDER BY c.class_level";
    
    $result = $conn->query($sql);
    $classes = [];
    
    while ($row = $result->fetch_assoc()) {
        $classes[] = $row;
    }
    
    return ['success' => true, 'data' => $classes];
}

function createClass($conn, $data) {
    $stmt = $conn->prepare("INSERT INTO classes (class_name, class_level, capacity, description) VALUES (?, ?, ?, ?)");
    $stmt->bind_param("siis", $data['class_name'], $data['class_level'], $data['capacity'], $data['description']);
    
    if ($stmt->execute()) {
        return ['success' => true, 'message' => 'Class created successfully', 'id' => $conn->insert_id];
    }
    return ['success' => false, 'message' => 'Failed to create class'];
}

function updateClass($conn, $data) {
    $stmt = $conn->prepare("UPDATE classes SET class_name = ?, class_level = ?, capacity = ?, description = ? WHERE id = ?");
    $stmt->bind_param("siisi", $data['class_name'], $data['class_level'], $data['capacity'], $data['description'], $data['id']);
    
    if ($stmt->execute()) {
        return ['success' => true, 'message' => 'Class updated successfully'];
    }
    return ['success' => false, 'message' => 'Failed to update class'];
}

function deleteClass($conn, $id) {
    $stmt = $conn->prepare("UPDATE classes SET is_active = 0 WHERE id = ?");
    $stmt->bind_param("i", $id);
    
    if ($stmt->execute()) {
        return ['success' => true, 'message' => 'Class deleted successfully'];
    }
    return ['success' => false, 'message' => 'Failed to delete class'];
}

function getClassStudents($conn, $classId) {
    if (!$classId) {
        return ['success' => false, 'message' => 'Class ID is required'];
    }
    
    // Get class name
    $stmt = $conn->prepare("SELECT class_name FROM classes WHERE id = ?");
    $stmt->bind_param("i", $classId);
    $stmt->execute();
    $classResult = $stmt->get_result()->fetch_assoc();
    
    if (!$classResult) {
        return ['success' => false, 'message' => 'Class not found'];
    }
    
    $className = $classResult['class_name'];
    
    // Get students by class_id or class name (for backward compatibility)
    $sql = "SELECT id, student_id as studentId, 
            CONCAT(first_name, ' ', last_name) as name,
            class, class_id, gender, date_of_birth as dateOfBirth,
            father_name as parentName, phone_number as parentContact,
            address, status, created_at as createdAt
            FROM students 
            WHERE (class_id = ? OR class = ?) 
            AND status = 'active'
            ORDER BY name";
    
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("is", $classId, $className);
    $stmt->execute();
    $result = $stmt->get_result();
    
    $students = [];
    while ($row = $result->fetch_assoc()) {
        $students[] = $row;
    }
    
    return ['success' => true, 'data' => $students, 'count' => count($students)];
}

// ============================================
// SUBJECTS MANAGEMENT
// ============================================
function handleSubjects($conn, $method, $data) {
    try {
        // Create subjects table if it doesn't exist
        $conn->exec("CREATE TABLE IF NOT EXISTS subjects (
            id INT AUTO_INCREMENT PRIMARY KEY,
            subject_name VARCHAR(100) NOT NULL,
            subject_code VARCHAR(20),
            description TEXT,
            is_active TINYINT(1) DEFAULT 1,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
        )");

        switch($method) {
            case 'GET':
                $query = "SELECT * FROM subjects WHERE is_active = 1 ORDER BY subject_name ASC";
                $stmt = $conn->prepare($query);
                $stmt->execute();
                $subjects = $stmt->fetchAll();
                return ['success' => true, 'data' => $subjects];

            case 'POST':
                $stmt = $conn->prepare("INSERT INTO subjects (subject_name, subject_code, description) VALUES (?, ?, ?)");
                $stmt->execute([
                    $data['subject_name'] ?? $data['name'] ?? '',
                    $data['subject_code'] ?? '',
                    $data['description'] ?? ''
                ]);
                return ['success' => true, 'message' => 'Subject created', 'id' => $conn->lastInsertId()];

            case 'PUT':
                $stmt = $conn->prepare("UPDATE subjects SET subject_name = ?, subject_code = ?, description = ? WHERE id = ?");
                $stmt->execute([
                    $data['subject_name'] ?? $data['name'] ?? '',
                    $data['subject_code'] ?? '',
                    $data['description'] ?? '',
                    $data['id']
                ]);
                return ['success' => true, 'message' => 'Subject updated'];

            case 'DELETE':
                $id = $data['id'] ?? null;
                if (!$id) {
                    throw new Exception('Subject ID is required');
                }
                $stmt = $conn->prepare("UPDATE subjects SET is_active = 0 WHERE id = ?");
                $stmt->execute([$id]);
                return ['success' => true, 'message' => 'Subject deleted'];

            default:
                return ['success' => false, 'message' => 'Method not allowed'];
        }
    } catch (Exception $e) {
        error_log("Subjects handler error: " . $e->getMessage());
        return ['success' => false, 'message' => $e->getMessage()];
    }
}

// ============================================
// CLASS-SUBJECTS MANAGEMENT
// ============================================

function handleClassSubjects($conn, $method, $data) {
    switch ($method) {
        case 'GET':
            return getClassSubjects($conn, $data['class_id'] ?? null);
        case 'POST':
            return assignSubjectToClass($conn, $data);
        case 'DELETE':
            return removeSubjectFromClass($conn, $data['id'] ?? null);
        default:
            return ['success' => false, 'message' => 'Invalid method'];
    }
}

function getClassSubjects($conn, $classId) {
    $sql = "SELECT cs.*, c.class_name, s.subject_name, s.subject_code 
            FROM class_subjects cs
            JOIN classes c ON cs.class_id = c.id
            JOIN subjects s ON cs.subject_id = s.id
            WHERE cs.is_active = 1";
    
    if ($classId) {
        $sql .= " AND cs.class_id = ?";
        $stmt = $conn->prepare($sql);
        $stmt->bind_param("i", $classId);
        $stmt->execute();
        $result = $stmt->get_result();
    } else {
        $result = $conn->query($sql);
    }
    
    $classSubjects = [];
    while ($row = $result->fetch_assoc()) {
        $classSubjects[] = $row;
    }
    
    return ['success' => true, 'data' => $classSubjects];
}

function assignSubjectToClass($conn, $data) {
    $stmt = $conn->prepare("INSERT INTO class_subjects (class_id, subject_id, pass_mark, credit_hours) VALUES (?, ?, ?, ?)");
    $stmt->bind_param("iidi", $data['class_id'], $data['subject_id'], $data['pass_mark'], $data['credit_hours']);
    
    if ($stmt->execute()) {
        return ['success' => true, 'message' => 'Subject assigned to class successfully', 'id' => $conn->insert_id];
    }
    return ['success' => false, 'message' => 'Failed to assign subject'];
}

function removeSubjectFromClass($conn, $id) {
    $stmt = $conn->prepare("UPDATE class_subjects SET is_active = 0 WHERE id = ?");
    $stmt->bind_param("i", $id);
    
    if ($stmt->execute()) {
        return ['success' => true, 'message' => 'Subject removed from class'];
    }
    return ['success' => false, 'message' => 'Failed to remove subject'];
}

// ============================================
// ACADEMIC TERMS
// ============================================

function handleTerms($conn, $method, $data) {
    switch ($method) {
        case 'GET':
            return getTerms($conn);
        case 'POST':
            return createTerm($conn, $data);
        case 'PUT':
            return updateTerm($conn, $data);
        default:
            return ['success' => false, 'message' => 'Invalid method'];
    }
}

function getTerms($conn) {
    $sql = "SELECT * FROM academic_terms ORDER BY academic_year DESC, start_date DESC";
    $result = $conn->query($sql);
    $terms = [];
    
    while ($row = $result->fetch_assoc()) {
        $terms[] = $row;
    }
    
    return ['success' => true, 'data' => $terms];
}

function getCurrentTerm($conn) {
    $sql = "SELECT * FROM academic_terms WHERE is_current = 1 LIMIT 1";
    $result = $conn->query($sql);
    
    if ($row = $result->fetch_assoc()) {
        return ['success' => true, 'data' => $row];
    }
    return ['success' => false, 'message' => 'No current term set'];
}

function createTerm($conn, $data) {
    // If this is set as current, unset all others
    if ($data['is_current'] ?? false) {
        $conn->query("UPDATE academic_terms SET is_current = 0");
    }
    
    $stmt = $conn->prepare("INSERT INTO academic_terms (term_name, academic_year, start_date, end_date, is_current) VALUES (?, ?, ?, ?, ?)");
    $isCurrent = $data['is_current'] ? 1 : 0;
    $stmt->bind_param("ssssi", $data['term_name'], $data['academic_year'], $data['start_date'], $data['end_date'], $isCurrent);
    
    if ($stmt->execute()) {
        return ['success' => true, 'message' => 'Term created successfully', 'id' => $conn->insert_id];
    }
    return ['success' => false, 'message' => 'Failed to create term'];
}

function updateTerm($conn, $data) {
    if ($data['is_current'] ?? false) {
        $conn->query("UPDATE academic_terms SET is_current = 0");
    }
    
    $stmt = $conn->prepare("UPDATE academic_terms SET term_name = ?, academic_year = ?, start_date = ?, end_date = ?, is_current = ? WHERE id = ?");
    $isCurrent = $data['is_current'] ? 1 : 0;
    $stmt->bind_param("ssssii", $data['term_name'], $data['academic_year'], $data['start_date'], $data['end_date'], $isCurrent, $data['id']);
    
    if ($stmt->execute()) {
        return ['success' => true, 'message' => 'Term updated successfully'];
    }
    return ['success' => false, 'message' => 'Failed to update term'];
}

// ============================================
// ASSESSMENTS
// ============================================

function handleAssessments($conn, $method, $data) {
    switch ($method) {
        case 'GET':
            return getAssessments($conn);
        case 'POST':
            return createAssessment($conn, $data);
        case 'PUT':
            return updateAssessment($conn, $data);
        case 'DELETE':
            return deleteAssessment($conn, $data['id'] ?? null);
        default:
            return ['success' => false, 'message' => 'Invalid method'];
    }
}

function getAssessments($conn) {
    $sql = "SELECT a.*, t.term_name, c.class_name, s.subject_name, at.type_name
            FROM assessments a
            JOIN academic_terms t ON a.term_id = t.id
            JOIN classes c ON a.class_id = c.id
            JOIN subjects s ON a.subject_id = s.id
            JOIN assessment_types at ON a.assessment_type_id = at.id
            ORDER BY a.assessment_date DESC";
    
    $result = $conn->query($sql);
    $assessments = [];
    
    while ($row = $result->fetch_assoc()) {
        $assessments[] = $row;
    }
    
    return ['success' => true, 'data' => $assessments];
}

function getAssessmentsByClass($conn, $data) {
    $sql = "SELECT a.*, t.term_name, s.subject_name, at.type_name,
            (SELECT COUNT(*) FROM student_scores WHERE assessment_id = a.id) as scores_recorded
            FROM assessments a
            JOIN academic_terms t ON a.term_id = t.id
            JOIN subjects s ON a.subject_id = s.id
            JOIN assessment_types at ON a.assessment_type_id = at.id
            WHERE a.class_id = ?";
    
    if (isset($data['term_id'])) {
        $sql .= " AND a.term_id = ?";
    }
    
    $sql .= " ORDER BY a.assessment_date DESC";
    
    $stmt = $conn->prepare($sql);
    if (isset($data['term_id'])) {
        $stmt->bind_param("ii", $data['class_id'], $data['term_id']);
    } else {
        $stmt->bind_param("i", $data['class_id']);
    }
    
    $stmt->execute();
    $result = $stmt->get_result();
    $assessments = [];
    
    while ($row = $result->fetch_assoc()) {
        $assessments[] = $row;
    }
    
    return ['success' => true, 'data' => $assessments];
}

function createAssessment($conn, $data) {
    $stmt = $conn->prepare("INSERT INTO assessments (term_id, class_id, subject_id, assessment_type_id, assessment_name, assessment_date, max_score, weight_percentage, description, created_by) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
    $stmt->bind_param("iiiissdds i", 
        $data['term_id'], 
        $data['class_id'], 
        $data['subject_id'], 
        $data['assessment_type_id'], 
        $data['assessment_name'], 
        $data['assessment_date'], 
        $data['max_score'], 
        $data['weight_percentage'], 
        $data['description'],
        $data['created_by']
    );
    
    if ($stmt->execute()) {
        return ['success' => true, 'message' => 'Assessment created successfully', 'id' => $conn->insert_id];
    }
    return ['success' => false, 'message' => 'Failed to create assessment'];
}

function updateAssessment($conn, $data) {
    $stmt = $conn->prepare("UPDATE assessments SET assessment_name = ?, assessment_date = ?, max_score = ?, weight_percentage = ?, description = ? WHERE id = ?");
    $stmt->bind_param("ssddsi", $data['assessment_name'], $data['assessment_date'], $data['max_score'], $data['weight_percentage'], $data['description'], $data['id']);
    
    if ($stmt->execute()) {
        return ['success' => true, 'message' => 'Assessment updated successfully'];
    }
    return ['success' => false, 'message' => 'Failed to update assessment'];
}

function deleteAssessment($conn, $id) {
    $stmt = $conn->prepare("DELETE FROM assessments WHERE id = ?");
    $stmt->bind_param("i", $id);
    
    if ($stmt->execute()) {
        return ['success' => true, 'message' => 'Assessment deleted successfully'];
    }
    return ['success' => false, 'message' => 'Failed to delete assessment'];
}

// ============================================
// SCORES
// ============================================

function handleAcademicScores($conn, $method, $data) {
    switch ($method) {
        case 'POST':
            return recordScore($conn, $data);
        case 'PUT':
            return updateScore($conn, $data);
        case 'DELETE':
            return deleteScore($conn, $data['id'] ?? null);
        default:
            return ['success' => false, 'message' => 'Invalid method'];
    }
}

function getScoresByAssessment($conn, $assessmentId) {
    $sql = "SELECT ss.*, s.name as student_name, s.studentId
            FROM student_scores ss
            JOIN students s ON ss.student_id = s.id
            WHERE ss.assessment_id = ?
            ORDER BY s.name";
    
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("i", $assessmentId);
    $stmt->execute();
    $result = $stmt->get_result();
    $scores = [];
    
    while ($row = $result->fetch_assoc()) {
        $scores[] = $row;
    }
    
    return ['success' => true, 'data' => $scores];
}

function getScoresByStudent($conn, $studentId, $termId) {
    $sql = "SELECT ss.*, a.assessment_name, a.max_score, s.subject_name, at.type_name, a.assessment_date
            FROM student_scores ss
            JOIN assessments a ON ss.assessment_id = a.id
            JOIN subjects s ON a.subject_id = s.id
            JOIN assessment_types at ON a.assessment_type_id = at.id
            WHERE ss.student_id = ?";
    
    if ($termId) {
        $sql .= " AND a.term_id = ?";
    }
    
    $sql .= " ORDER BY a.assessment_date DESC";
    
    $stmt = $conn->prepare($sql);
    if ($termId) {
        $stmt->bind_param("ii", $studentId, $termId);
    } else {
        $stmt->bind_param("i", $studentId);
    }
    
    $stmt->execute();
    $result = $stmt->get_result();
    $scores = [];
    
    while ($row = $result->fetch_assoc()) {
        $scores[] = $row;
    }
    
    return ['success' => true, 'data' => $scores];
}

function recordScore($conn, $data) {
    $stmt = $conn->prepare("INSERT INTO student_scores (assessment_id, student_id, score, remarks, recorded_by) VALUES (?, ?, ?, ?, ?) ON DUPLICATE KEY UPDATE score = ?, remarks = ?");
    $stmt->bind_param("iidsids", 
        $data['assessment_id'], 
        $data['student_id'], 
        $data['score'], 
        $data['remarks'],
        $data['recorded_by'],
        $data['score'],
        $data['remarks']
    );
    
    if ($stmt->execute()) {
        return ['success' => true, 'message' => 'Score recorded successfully'];
    }
    return ['success' => false, 'message' => 'Failed to record score'];
}

function updateScore($conn, $data) {
    $stmt = $conn->prepare("UPDATE student_scores SET score = ?, remarks = ? WHERE id = ?");
    $stmt->bind_param("dsi", $data['score'], $data['remarks'], $data['id']);
    
    if ($stmt->execute()) {
        return ['success' => true, 'message' => 'Score updated successfully'];
    }
    return ['success' => false, 'message' => 'Failed to update score'];
}

function deleteScore($conn, $id) {
    $stmt = $conn->prepare("DELETE FROM student_scores WHERE id = ?");
    $stmt->bind_param("i", $id);
    
    if ($stmt->execute()) {
        return ['success' => true, 'message' => 'Score deleted successfully'];
    }
    return ['success' => false, 'message' => 'Failed to delete score'];
}

// ============================================
// ATTENDANCE
// ============================================

function handleAttendance($conn, $method, $data) {
    switch ($method) {
        case 'POST':
            return recordAttendance($conn, $data);
        case 'PUT':
            return updateAttendance($conn, $data);
        default:
            return ['success' => false, 'message' => 'Invalid method'];
    }
}

function getAttendanceByClass($conn, $data) {
    $sql = "SELECT a.*, s.name as student_name, s.studentId
            FROM attendance a
            JOIN students s ON a.student_id = s.id
            WHERE a.class_id = ? AND a.attendance_date = ?
            ORDER BY s.name";
    
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("is", $data['class_id'], $data['date']);
    $stmt->execute();
    $result = $stmt->get_result();
    $attendance = [];
    
    while ($row = $result->fetch_assoc()) {
        $attendance[] = $row;
    }
    
    return ['success' => true, 'data' => $attendance];
}

function getAttendanceByStudent($conn, $studentId, $data) {
    $sql = "SELECT * FROM attendance WHERE student_id = ?";
    
    if (isset($data['start_date']) && isset($data['end_date'])) {
        $sql .= " AND attendance_date BETWEEN ? AND ?";
        $stmt = $conn->prepare($sql);
        $stmt->bind_param("iss", $studentId, $data['start_date'], $data['end_date']);
    } else {
        $stmt = $conn->prepare($sql);
        $stmt->bind_param("i", $studentId);
    }
    
    $stmt->execute();
    $result = $stmt->get_result();
    $attendance = [];
    
    while ($row = $result->fetch_assoc()) {
        $attendance[] = $row;
    }
    
    return ['success' => true, 'data' => $attendance];
}

function getAttendanceSummary($conn, $data) {
    $sql = "SELECT 
            COUNT(*) as total_days,
            SUM(CASE WHEN status = 'present' THEN 1 ELSE 0 END) as present_days,
            SUM(CASE WHEN status = 'absent' THEN 1 ELSE 0 END) as absent_days,
            SUM(CASE WHEN status = 'late' THEN 1 ELSE 0 END) as late_days,
            SUM(CASE WHEN status = 'excused' THEN 1 ELSE 0 END) as excused_days
            FROM attendance 
            WHERE student_id = ?";
    
    if (isset($data['start_date']) && isset($data['end_date'])) {
        $sql .= " AND attendance_date BETWEEN ? AND ?";
        $stmt = $conn->prepare($sql);
        $stmt->bind_param("iss", $data['student_id'], $data['start_date'], $data['end_date']);
    } else {
        $stmt = $conn->prepare($sql);
        $stmt->bind_param("i", $data['student_id']);
    }
    
    $stmt->execute();
    $result = $stmt->get_result();
    $summary = $result->fetch_assoc();
    
    // Calculate percentage
    if ($summary['total_days'] > 0) {
        $summary['attendance_percentage'] = round(($summary['present_days'] / $summary['total_days']) * 100, 2);
    } else {
        $summary['attendance_percentage'] = 0;
    }
    
    return ['success' => true, 'data' => $summary];
}

function recordAttendance($conn, $data) {
    $stmt = $conn->prepare("INSERT INTO attendance (student_id, class_id, attendance_date, status, arrival_time, remarks, recorded_by) VALUES (?, ?, ?, ?, ?, ?, ?) ON DUPLICATE KEY UPDATE status = ?, arrival_time = ?, remarks = ?");
    $stmt->bind_param("iissssisss", 
        $data['student_id'], 
        $data['class_id'], 
        $data['attendance_date'], 
        $data['status'],
        $data['arrival_time'],
        $data['remarks'],
        $data['recorded_by'],
        $data['status'],
        $data['arrival_time'],
        $data['remarks']
    );
    
    if ($stmt->execute()) {
        return ['success' => true, 'message' => 'Attendance recorded successfully'];
    }
    return ['success' => false, 'message' => 'Failed to record attendance'];
}

function updateAttendance($conn, $data) {
    $stmt = $conn->prepare("UPDATE attendance SET status = ?, arrival_time = ?, remarks = ? WHERE id = ?");
    $stmt->bind_param("sssi", $data['status'], $data['arrival_time'], $data['remarks'], $data['id']);
    
    if ($stmt->execute()) {
        return ['success' => true, 'message' => 'Attendance updated successfully'];
    }
    return ['success' => false, 'message' => 'Failed to update attendance'];
}

// ============================================
// REPORT CARDS
// ============================================

function handleReportCards($conn, $method, $data) {
    switch ($method) {
        case 'GET':
            return getReportCards($conn, $data);
        case 'POST':
            return generateReportCard($conn, $data);
        case 'PUT':
            return updateReportCard($conn, $data);
        default:
            return ['success' => false, 'message' => 'Invalid method'];
    }
}

function getReportCards($conn, $data) {
    $sql = "SELECT rc.*, s.name as student_name, s.studentId, t.term_name, c.class_name
            FROM report_cards rc
            JOIN students s ON rc.student_id = s.id
            JOIN academic_terms t ON rc.term_id = t.id
            JOIN classes c ON rc.class_id = c.id";
    
    $conditions = [];
    $params = [];
    $types = "";
    
    if (isset($data['student_id'])) {
        $conditions[] = "rc.student_id = ?";
        $params[] = $data['student_id'];
        $types .= "i";
    }
    
    if (isset($data['term_id'])) {
        $conditions[] = "rc.term_id = ?";
        $params[] = $data['term_id'];
        $types .= "i";
    }
    
    if (isset($data['class_id'])) {
        $conditions[] = "rc.class_id = ?";
        $params[] = $data['class_id'];
        $types .= "i";
    }
    
    if (!empty($conditions)) {
        $sql .= " WHERE " . implode(" AND ", $conditions);
    }
    
    $sql .= " ORDER BY rc.generated_at DESC";
    
    if (!empty($params)) {
        $stmt = $conn->prepare($sql);
        $stmt->bind_param($types, ...$params);
        $stmt->execute();
        $result = $stmt->get_result();
    } else {
        $result = $conn->query($sql);
    }
    
    $reportCards = [];
    while ($row = $result->fetch_assoc()) {
        $reportCards[] = $row;
    }
    
    return ['success' => true, 'data' => $reportCards];
}

function getReportCardByStudent($conn, $studentId, $termId) {
    // Get report card
    $sql = "SELECT rc.*, s.name as student_name, s.studentId, s.class, t.term_name, t.academic_year, c.class_name
            FROM report_cards rc
            JOIN students s ON rc.student_id = s.id
            JOIN academic_terms t ON rc.term_id = t.id
            JOIN classes c ON rc.class_id = c.id
            WHERE rc.student_id = ? AND rc.term_id = ?";
    
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("ii", $studentId, $termId);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($reportCard = $result->fetch_assoc()) {
        // Get subject details
        $sql = "SELECT rcd.*, s.subject_name, s.subject_code
                FROM report_card_details rcd
                JOIN subjects s ON rcd.subject_id = s.id
                WHERE rcd.report_card_id = ?
                ORDER BY s.subject_name";
        
        $stmt = $conn->prepare($sql);
        $stmt->bind_param("i", $reportCard['id']);
        $stmt->execute();
        $result = $stmt->get_result();
        
        $subjects = [];
        while ($row = $result->fetch_assoc()) {
            $subjects[] = $row;
        }
        
        $reportCard['subjects'] = $subjects;
        
        return ['success' => true, 'data' => $reportCard];
    }
    
    return ['success' => false, 'message' => 'Report card not found'];
}

function generateReportCard($conn, $data) {
    $studentId = $data['student_id'];
    $termId = $data['term_id'];
    $generatedBy = $data['generated_by'];
    
    // Get student info
    $stmt = $conn->prepare("SELECT * FROM students WHERE id = ?");
    $stmt->bind_param("i", $studentId);
    $stmt->execute();
    $student = $stmt->get_result()->fetch_assoc();
    
    if (!$student) {
        return ['success' => false, 'message' => 'Student not found'];
    }
    
    // Get class ID
    $stmt = $conn->prepare("SELECT id FROM classes WHERE class_name = ?");
    $stmt->bind_param("s", $student['class']);
    $stmt->execute();
    $classResult = $stmt->get_result()->fetch_assoc();
    $classId = $classResult['id'];
    
    // Get all assessments for this term and class
    $sql = "SELECT a.*, s.subject_id
            FROM assessments a
            JOIN class_subjects cs ON a.class_id = cs.class_id AND a.subject_id = cs.subject_id
            JOIN subjects s ON a.subject_id = s.id
            WHERE a.term_id = ? AND a.class_id = ?";
    
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("ii", $termId, $classId);
    $stmt->execute();
    $assessments = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    
    // Calculate scores per subject
    $subjectScores = [];
    
    foreach ($assessments as $assessment) {
        $subjectId = $assessment['subject_id'];
        
        // Get student's score for this assessment
        $stmt = $conn->prepare("SELECT score FROM student_scores WHERE assessment_id = ? AND student_id = ?");
        $stmt->bind_param("ii", $assessment['id'], $studentId);
        $stmt->execute();
        $scoreResult = $stmt->get_result()->fetch_assoc();
        
        if ($scoreResult) {
            $weightedScore = ($scoreResult['score'] / $assessment['max_score']) * $assessment['weight_percentage'];
            
            if (!isset($subjectScores[$subjectId])) {
                $subjectScores[$subjectId] = [
                    'total_weighted' => 0,
                    'total_weight' => 0
                ];
            }
            
            $subjectScores[$subjectId]['total_weighted'] += $weightedScore;
            $subjectScores[$subjectId]['total_weight'] += $assessment['weight_percentage'];
        }
    }
    
    // Calculate final scores and grades
    $totalScore = 0;
    $subjectCount = 0;
    
    $conn->begin_transaction();
    
    try {
        // Create or update report card
        $stmt = $conn->prepare("INSERT INTO report_cards (student_id, term_id, class_id, generated_by) VALUES (?, ?, ?, ?) ON DUPLICATE KEY UPDATE id = LAST_INSERT_ID(id)");
        $stmt->bind_param("iiii", $studentId, $termId, $classId, $generatedBy);
        $stmt->execute();
        $reportCardId = $conn->insert_id;
        
        // Delete existing details
        $stmt = $conn->prepare("DELETE FROM report_card_details WHERE report_card_id = ?");
        $stmt->bind_param("i", $reportCardId);
        $stmt->execute();
        
        // Insert subject details
        foreach ($subjectScores as $subjectId => $scores) {
            if ($scores['total_weight'] > 0) {
                $finalScore = ($scores['total_weighted'] / $scores['total_weight']) * 100;
                $grade = getGradeForScore($conn, $finalScore);
                
                // Get class average for this subject
                $classAvg = getClassAverageForSubject($conn, $termId, $classId, $subjectId);
                
                // Get subject position
                $position = getSubjectPosition($conn, $termId, $classId, $subjectId, $studentId, $finalScore);
                
                $stmt = $conn->prepare("INSERT INTO report_card_details (report_card_id, subject_id, student_score, grade, class_average, subject_position) VALUES (?, ?, ?, ?, ?, ?)");
                $stmt->bind_param("iidsdi", $reportCardId, $subjectId, $finalScore, $grade, $classAvg, $position);
                $stmt->execute();
                
                $totalScore += $finalScore;
                $subjectCount++;
            }
        }
        
        // Calculate overall statistics
        $averageScore = $subjectCount > 0 ? $totalScore / $subjectCount : 0;
        $overallGrade = getGradeForScore($conn, $averageScore);
        $classPosition = getOverallClassPosition($conn, $termId, $classId, $studentId, $averageScore);
        $totalStudents = getTotalStudentsInClass($conn, $classId);
        
        // Get attendance summary
        $attendanceSummary = getAttendanceSummaryForTerm($conn, $studentId, $termId);
        
        // Update report card with calculated values
        $stmt = $conn->prepare("UPDATE report_cards SET total_score = ?, average_score = ?, grade = ?, class_position = ?, total_students = ?, attendance_percentage = ?, days_present = ?, days_absent = ? WHERE id = ?");
        $stmt->bind_param("ddsiiidii", 
            $totalScore, 
            $averageScore, 
            $overallGrade, 
            $classPosition, 
            $totalStudents,
            $attendanceSummary['attendance_percentage'],
            $attendanceSummary['present_days'],
            $attendanceSummary['absent_days'],
            $reportCardId
        );
        $stmt->execute();
        
        $conn->commit();
        
        return ['success' => true, 'message' => 'Report card generated successfully', 'report_card_id' => $reportCardId];
        
    } catch (Exception $e) {
        $conn->rollback();
        return ['success' => false, 'message' => 'Failed to generate report card: ' . $e->getMessage()];
    }
}

function updateReportCard($conn, $data) {
    $stmt = $conn->prepare("UPDATE report_cards SET teacher_comment = ?, principal_comment = ?, is_published = ? WHERE id = ?");
    $isPublished = $data['is_published'] ? 1 : 0;
    $stmt->bind_param("ssii", $data['teacher_comment'], $data['principal_comment'], $isPublished, $data['id']);
    
    if ($stmt->execute()) {
        if ($isPublished) {
            $stmt = $conn->prepare("UPDATE report_cards SET published_at = NOW() WHERE id = ?");
            $stmt->bind_param("i", $data['id']);
            $stmt->execute();
        }
        return ['success' => true, 'message' => 'Report card updated successfully'];
    }
    return ['success' => false, 'message' => 'Failed to update report card'];
}

// Helper functions
function getGradeForScore($conn, $score) {
    $stmt = $conn->prepare("SELECT grade FROM grading_scale WHERE ? BETWEEN min_score AND max_score AND is_active = 1 LIMIT 1");
    $stmt->bind_param("d", $score);
    $stmt->execute();
    $result = $stmt->get_result()->fetch_assoc();
    return $result ? $result['grade'] : 'N/A';
}

function getClassAverageForSubject($conn, $termId, $classId, $subjectId) {
    // This is a simplified version - you'd need to calculate weighted averages properly
    $sql = "SELECT AVG(ss.score) as avg_score
            FROM student_scores ss
            JOIN assessments a ON ss.assessment_id = a.id
            WHERE a.term_id = ? AND a.class_id = ? AND a.subject_id = ?";
    
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("iii", $termId, $classId, $subjectId);
    $stmt->execute();
    $result = $stmt->get_result()->fetch_assoc();
    return $result['avg_score'] ?? 0;
}

function getSubjectPosition($conn, $termId, $classId, $subjectId, $studentId, $studentScore) {
    // Count how many students scored higher
    $sql = "SELECT COUNT(DISTINCT ss.student_id) + 1 as position
            FROM student_scores ss
            JOIN assessments a ON ss.assessment_id = a.id
            WHERE a.term_id = ? AND a.class_id = ? AND a.subject_id = ? 
            AND ss.score > ?";
    
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("iiid", $termId, $classId, $subjectId, $studentScore);
    $stmt->execute();
    $result = $stmt->get_result()->fetch_assoc();
    return $result['position'] ?? 0;
}

function getOverallClassPosition($conn, $termId, $classId, $studentId, $averageScore) {
    // Simplified - count students with higher average
    $sql = "SELECT COUNT(*) + 1 as position
            FROM report_cards
            WHERE term_id = ? AND class_id = ? AND average_score > ?";
    
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("iid", $termId, $classId, $averageScore);
    $stmt->execute();
    $result = $stmt->get_result()->fetch_assoc();
    return $result['position'] ?? 1;
}

function getTotalStudentsInClass($conn, $classId) {
    $stmt = $conn->prepare("SELECT COUNT(*) as total FROM students WHERE class = (SELECT class_name FROM classes WHERE id = ?) AND status = 'active'");
    $stmt->bind_param("i", $classId);
    $stmt->execute();
    $result = $stmt->get_result()->fetch_assoc();
    return $result['total'] ?? 0;
}

function getAttendanceSummaryForTerm($conn, $studentId, $termId) {
    // Get term dates
    $stmt = $conn->prepare("SELECT start_date, end_date FROM academic_terms WHERE id = ?");
    $stmt->bind_param("i", $termId);
    $stmt->execute();
    $term = $stmt->get_result()->fetch_assoc();
    
    if (!$term) {
        return ['attendance_percentage' => 0, 'present_days' => 0, 'absent_days' => 0];
    }
    
    $sql = "SELECT 
            COUNT(*) as total_days,
            SUM(CASE WHEN status = 'present' THEN 1 ELSE 0 END) as present_days,
            SUM(CASE WHEN status = 'absent' THEN 1 ELSE 0 END) as absent_days
            FROM attendance 
            WHERE student_id = ? AND attendance_date BETWEEN ? AND ?";
    
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("iss", $studentId, $term['start_date'], $term['end_date']);
    $stmt->execute();
    $summary = $stmt->get_result()->fetch_assoc();
    
    $summary['attendance_percentage'] = $summary['total_days'] > 0 
        ? round(($summary['present_days'] / $summary['total_days']) * 100, 2) 
        : 0;
    
    return $summary;
}

// ============================================
// UTILITY FUNCTIONS
// ============================================

function getGradingScale($conn) {
    $sql = "SELECT * FROM grading_scale WHERE is_active = 1 ORDER BY min_score DESC";
    $result = $conn->query($sql);
    $scale = [];
    
    while ($row = $result->fetch_assoc()) {
        $scale[] = $row;
    }
    
    return ['success' => true, 'data' => $scale];
}

function getAssessmentTypes($conn) {
    $sql = "SELECT * FROM assessment_types WHERE is_active = 1 ORDER BY type_name";
    $result = $conn->query($sql);
    $types = [];
    
    while ($row = $result->fetch_assoc()) {
        $types[] = $row;
    }
    
    return ['success' => true, 'data' => $types];
}
